--[[
------------------------------------------------------
          UNESCO - Inria - Software Heritage
------------------------------------------------------

This model was made to celebrate the signature of the partnership
between UNESCO and Inria on the preservation and sharing of our
software heritage. Its shape is procedurally generated by the script
below from a unique seed number.

Two versions of this sculpture were specially 3D printed and offered
to the French President, Mr. Francois Hollande, and the Director
General of Unesco, Ms. Irina Bokova, by Inria's CEO, Mr. Antoine
Petit, during the ceremony that took place on April 3rd 2017, in
Paris, at Unesco's headquarters.

Seven additional sculptures were offered to the sponsors of Software
Heritage: Microsoft, Intel, Societe Generale, Huawei, Dans, Disi,
Nokia Bell Labs.

HOW TO
----------

Download IceSL from http://shapeforge.loria.fr/icesl
Install IceSL, and load this script.

You can now modify the script to create your own version!
For instance simply change the value of 'seed' below.
You can also displace the logo using the 'sh_pos_x' variable, and
see how the branches adapt.

To output a mesh change the value of 'do_mesh' from false to true.

Beware this will take a few seconds and output a large detailed 
mesh, ~ 200 MB. The mesh is output as 'sculpture.stl'
To remesh the model into a lighter triangle mesh we recommend
Graphite https://gforge.inria.fr/frs/?group_id=1465

------------------------------------------------------
Released under the Creative Commons License CC-BY-SA
Design (c) Sylvain Lefebvre - Inria
------------------------------------------------------
]]

seed = 101 -- any integer
sh_pos_x = 0 -- try values between -15 and 15
do_mesh = false -- set to true to output a mesh (beware: this might take a few seconds and will produce a large output)

--------------------------------------------

math.randomseed(seed)

------ Software Heritage logo

cc = ccube(8,1.2,32)
atom = union{
rotate(0,0,65)*cc,
translate(3.6,0,0)*rotate(0,0,-65)*cc,
translate(1.6,-6.7,0)*rotate(0,0,163)*ccube(11,1.2,32),
translate(0,-13.5,0)*rotate(0,0,-65)*cc,
translate(0,-13.5,0)*translate(3.7,0,0)*rotate(0,0,65)*cc,
}
logo = {}
for a = 1,360,360/8 do
	table.insert(logo,
      rotate(0,0,a)
     *translate(10.3,0,0)
     *rotate(0,0,98)*atom)
end
sh = difference(
  cylinder(25,2),
  scale(0.8,0.8,1)*union(logo)
)

------ Tree

-- Bezier spline interpolation
function bezier(u,p0,p1,p2,p3)
  local tensil = 0.125
  local t_p1 = p1 + (p2 - p0)*tensil
  local t_p2 = p2 - (p3 - p1)*tensil
  local u2 = u*u
  local u3 = u2*u
  local c = 3.0 * (t_p1 - p1)
  local b = 3.0 * (t_p2 - t_p1) - c
  local a = p2 - p1 - c - b
  return a * u3 + b * u2 + c * u + p1
end

-- interpolates a path using Bezier spline
function smooth(tbl)
  if not do_mesh then -- skip if not producing the final mesh
    return tbl
  end
  local num_start = #tbl
  local all = {}
  local num = #tbl
  local p0,p1,p2,p3
  for i,p in ipairs(tbl) do
    if i == 1 then
       p0 = p
       p1 = p
       p2 = p
       p3 = p
    else
       local pb = bezier(0.5,p0,p1,p2,p3)
       table.insert(all,p1)
       table.insert(all,pb)
    end
    p0 = p1
    p1 = p2
    p2 = p3
    p3 = p
  end
  return all
end

-- generates a path geometry using cones and spheres
function path(tbl,r_base)
    local r_tip = 1
    local r = r_base
    local prev
    local r_prev
    local all = {}
    local num = #tbl
    for i,p in ipairs(tbl) do
        if i == 1 then
          prev = p
          r_prev = r
        else
          if not do_mesh then
            r = math.max(r * 0.95,r_tip)
          else
            r = math.max(r * math.sqrt(0.95),r_tip)
          end
          table.insert(all,(cone(r_prev,r,prev,p)))
		  table.insert(all,(translate(prev)*sphere(r_prev)))
        end
        prev = p
        r_prev = r
    end
    return (merge(all))
end

-- perturbation function (noise)
function perturb(dir,p,seed)
  nrm = v(-dir.y,dir.x,0)
  return nrm * (
         4.0*sin(p.y * 2.4 + seed * 133)
      )
end

-- global variable storing all branches
branches = {}
T_branch = translate(0,0,0)

-- grows a branch, might spawn child branches
-- this is tracing particles, avoiding the spheres defined in bumps
function grow(seed,start,maindir,bumps,curl,len,branching,recurs,sub,curl_len)
  local branch = {}
  local prev = start
  local delta = v(0,0,0)
  if sub then
    for _,b in ipairs(sub) do
      table.insert(branch,b)
	  delta = b - start
    end
  else
    table.insert(branch,start)
  end
  local step = 5
  local i = 0
  local curl_strength = 0.13
  if not recurs then
    recurs = 0
    curl_strength = 0.08
  end
  local curl_len = 24
  local curl_start = len - curl_len
  while i < curl_start + len do
    local dir = maindir + 0.10*perturb(maindir,prev,seed)
    local repulse = v(0,0,0)
    for _,b in ipairs(bumps) do
	   local d = b[1] - prev
	   d = d - dot(dir,d) * normalize(dir)
       local dir_bump = - normalize(d)
	   local l = length(b[1] - prev)
	   local strength_bump = math.max(0,b[2] - l)
	   repulse = repulse + 0.2 * dir_bump * strength_bump
    end
    repulse = repulse * v(1,1,0)
    repulse = repulse + 0.01*v(- prev.x,0,0)
    dir = dir + repulse
    local p = v(0,0,0)
    if i < curl_start then
      dir = normalize(dir)
      p = prev + step * dir
      table.insert(branch,p)
       if prev.x > 0 then curl = 1 else curl = -1 end
    else
	  -- curl tips
      local nrm = v(-delta.y,delta.x,0)
      p = prev 
        + step * delta / (curl_strength*(11 + i - curl_start))
        + curl * nrm + 0.3*repulse
      table.insert(branch,p)
    end

	local thres = 55
    if  -- prev.y < thres and p.y > thres 
	   prev.y > 15
	   and ((i+seed)%branching) == 0
	   -- and prev.y < 60
	   and i < len/3
	   and recurs == 0 then
         grow(seed,prev,
		         maindir,
				 bumps,-curl,
				 curl_len,
				 1,
				 1+recurs,
				 branch)
    end
	
    delta = normalize(p - prev)
	if recurs == 0 then
      curl_start = curl_start - 1.2*(1-delta.y*delta.y)
      curl_start = curl_start - 0.00003*(p.y*p.y)
	end
	prev = p
    i = i + 1
  end
  local radius_base = 7
  if recurs == 1 then
    radius_base = 5
  end
  table.insert(branches,T_branch * path(smooth(branch),radius_base))
end

------ generates the full tree

sh_pos = v(sh_pos_x,67,9) 
-- things to avoid
bumps = {}
table.insert(bumps,{sh_pos,40})

-- branches
num_branches = 6
for i = 0,num_branches-1 do
  a = 360 * i / num_branches
  r0 = 15
  r1 = 5
  pos = v(r0*cos(a),0,r1*sin(a))
  dir = -pos + v(0,20+math.random(0,20),0)
  dir = dir + 1*v(0,0,1.0 - 2.0*math.random())
  dir.z = dir.z * 0.7
  grow(seed+i,pos,normalize(dir),bumps,(i%2)*2-1,55,
      6 + math.floor(3*math.random()))
end
-- fixed branch below SH logo
T_branch = translate(sh_pos_x,0,0)
grow(43,v(0,0,0),v(-0.6,1.00,0),{},-1,29,100)
T_branch = translate(0,0,0)

-- base

base_h = 22*1.2
base_d_top  = 31*1.2
base_d_btm = 48*1.2
pts = {}
for i=0,5 do
  pt = v(base_d_top * cos(i*360/6),base_d_top * sin(i*360/6),base_h)
  table.insert(pts,pt)
end
pts_btm = {}
for i=0,5 do
  pt = v(base_d_btm * cos(i*360/6),base_d_btm * sin(i*360/6),0)
  table.insert(pts,pt)
  table.insert(pts_btm,pt)
end
base_btm = translate(0,0,-1.5)*linear_extrude(v(0,0,1.5),pts_btm)
tris = {
  v(0,1,6),v(1,7,6),
  v(1,2,7),v(2,8,7),
  v(2,3,8),v(3,9,8),
  v(3,4,9),v(4,10,9),
  v(4,5,10),v(5,11,10),
  v(5,0,11),v(0,6,11),
  v(0,2,1),v(0,3,2),v(0,4,3),v(0,5,4),
  v(6,7,8),v(6,8,9),v(6,9,10),v(6,10,11),
  }

base = merge{flip(polyhedron(pts,tris)),base_btm}

base_carve = intersection{
     translate(0,0,-8)*base,   
     translate(0,0,8)*base,
     translate(10,0,0)*base,
     translate(-10,0,0)*base,
     translate(0,10,0)*base,
     translate(0,-10,0)*base,
     }

base = difference(base,base_carve)

final = union{
 translate(sh_pos)*translate(-1,4,-8)*rotate(0,0,9)*scale(1.0,1.0,1)*sh,
 union(branches)
}

---------------------------------------------------
-- Text and logos
---------------------------------------------------

if not inria then
inria = merge{flip(linear_extrude_from_oriented(v(0,0,1),{v(29.55,5.58,0.00),v(30.21,5.02,0.00),v(30.79,4.50,0.00),v(31.29,4.01,0.00),v(31.72,3.56,0.00),v(32.09,3.15,0.00),v(32.39,2.77,0.00),v(32.63,2.42,0.00),v(32.81,2.11,0.00),v(32.94,1.83,0.00),v(33.02,1.58,0.00),v(33.06,1.35,0.00),v(33.05,1.16,0.00),v(33.00,0.99,0.00),v(32.91,0.85,0.00),v(32.79,0.73,0.00),v(32.64,0.64,0.00),v(32.47,0.56,0.00),v(32.27,0.51,0.00),v(31.82,0.47,0.00),v(31.31,0.49,0.00),v(30.79,0.57,0.00),v(30.27,0.71,0.00),v(29.78,0.89,0.00),v(29.35,1.11,0.00),v(29.01,1.34,0.00),v(28.49,1.88,0.00),v(28.03,2.37,0.00),v(27.62,2.84,0.00),v(27.27,3.27,0.00),v(26.97,3.66,0.00),v(26.72,4.02,0.00),v(26.51,4.36,0.00),v(26.35,4.66,0.00),v(26.22,4.93,0.00),v(26.14,5.18,0.00),v(26.09,5.40,0.00),v(26.08,5.59,0.00),v(26.10,5.76,0.00),v(26.15,5.90,0.00),v(26.23,6.02,0.00),v(26.34,6.12,0.00),v(26.46,6.20,0.00),v(26.61,6.26,0.00),v(26.96,6.32,0.00),v(27.36,6.32,0.00),v(27.80,6.26,0.00),v(28.26,6.14,0.00),v(28.71,5.99,0.00),v(29.15,5.80,0.00),v(29.55,5.58,0.00),v(29.55,5.58,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(146.96,38.88,0.00),v(147.15,38.96,0.00),v(147.32,38.95,0.00),v(147.46,38.86,0.00),v(147.57,38.70,0.00),v(147.64,38.48,0.00),v(147.68,38.20,0.00),v(147.68,37.88,0.00),v(147.64,37.52,0.00),v(147.56,37.14,0.00),v(147.44,36.75,0.00),v(147.27,36.34,0.00),v(147.06,35.94,0.00),v(146.81,35.56,0.00),v(146.51,35.20,0.00),v(146.15,34.86,0.00),v(145.75,34.57,0.00),v(145.35,34.34,0.00),v(144.94,34.13,0.00),v(144.50,33.94,0.00),v(144.05,33.77,0.00),v(143.08,33.48,0.00),v(142.05,33.27,0.00),v(140.95,33.12,0.00),v(139.80,33.03,0.00),v(138.60,33.00,0.00),v(137.34,33.02,0.00),v(136.05,33.10,0.00),v(134.72,33.22,0.00),v(133.35,33.38,0.00),v(131.96,33.58,0.00),v(130.55,33.81,0.00),v(129.12,34.08,0.00),v(126.23,34.68,0.00),v(123.35,35.36,0.00),v(120.49,36.07,0.00),v(117.72,36.80,0.00),v(116.38,37.15,0.00),v(115.07,37.50,0.00),v(113.80,37.83,0.00),v(112.58,38.14,0.00),v(111.41,38.43,0.00),v(110.29,38.69,0.00),v(109.24,38.92,0.00),v(108.25,39.12,0.00),v(107.33,39.28,0.00),v(106.49,39.40,0.00),v(106.20,39.43,0.00),v(105.94,39.44,0.00),v(105.70,39.44,0.00),v(105.48,39.43,0.00),v(105.29,39.40,0.00),v(105.12,39.36,0.00),v(104.97,39.30,0.00),v(104.85,39.23,0.00),v(104.75,39.14,0.00),v(104.67,39.04,0.00),v(104.62,38.92,0.00),v(104.58,38.79,0.00),v(104.58,38.64,0.00),v(104.59,38.48,0.00),v(104.63,38.30,0.00),v(104.69,38.10,0.00),v(104.85,37.69,0.00),v(105.05,37.24,0.00),v(105.29,36.77,0.00),v(105.56,36.27,0.00),v(106.17,35.24,0.00),v(106.82,34.19,0.00),v(107.46,33.17,0.00),v(107.77,32.69,0.00),v(108.05,32.24,0.00),v(108.30,31.82,0.00),v(108.52,31.45,0.00),v(108.70,31.11,0.00),v(108.84,30.84,0.00),v(108.93,30.56,0.00),v(109.00,30.23,0.00),v(109.04,29.85,0.00),v(109.04,29.43,0.00),v(108.99,28.97,0.00),v(108.88,28.50,0.00),v(108.71,28.03,0.00),v(108.47,27.55,0.00),v(108.14,27.10,0.00),v(107.73,26.66,0.00),v(107.21,26.27,0.00),v(106.92,26.09,0.00),v(106.59,25.92,0.00),v(106.24,25.77,0.00),v(105.86,25.63,0.00),v(105.44,25.51,0.00),v(105.00,25.41,0.00),v(104.52,25.33,0.00),v(104.00,25.27,0.00),v(103.45,25.23,0.00),v(102.87,25.22,0.00),v(101.89,25.25,0.00),v(100.92,25.33,0.00),v(99.97,25.47,0.00),v(99.02,25.65,0.00),v(98.10,25.88,0.00),v(97.19,26.16,0.00),v(96.30,26.47,0.00),v(95.44,26.83,0.00),v(94.59,27.22,0.00),v(93.77,27.64,0.00),v(92.97,28.10,0.00),v(92.20,28.58,0.00),v(91.45,29.09,0.00),v(90.74,29.62,0.00),v(90.05,30.17,0.00),v(89.40,30.74,0.00),v(88.76,31.28,0.00),v(88.06,31.88,0.00),v(87.30,32.52,0.00),v(86.50,33.18,0.00),v(85.65,33.86,0.00),v(84.78,34.54,0.00),v(82.97,35.87,0.00),v(82.04,36.50,0.00),v(81.12,37.08,0.00),v(80.21,37.60,0.00),v(79.32,38.05,0.00),v(78.45,38.43,0.00),v(77.61,38.71,0.00),v(77.20,38.81,0.00),v(76.81,38.88,0.00),v(76.43,38.93,0.00),v(76.06,38.95,0.00),v(75.89,38.94,0.00),v(75.73,38.91,0.00),v(75.49,38.81,0.00),v(75.33,38.65,0.00),v(75.25,38.43,0.00),v(75.24,38.15,0.00),v(75.29,37.83,0.00),v(75.41,37.46,0.00),v(75.59,37.05,0.00),v(75.82,36.61,0.00),v(76.10,36.13,0.00),v(76.42,35.63,0.00),v(76.78,35.10,0.00),v(77.17,34.56,0.00),v(77.59,34.00,0.00),v(78.49,32.85,0.00),v(79.45,31.69,0.00),v(80.43,30.56,0.00),v(81.38,29.50,0.00),v(81.83,29.00,0.00),v(82.26,28.53,0.00),v(82.67,28.09,0.00),v(83.05,27.68,0.00),v(83.39,27.32,0.00),v(83.69,27.01,0.00),v(83.95,26.74,0.00),v(84.15,26.53,0.00),v(84.31,26.38,0.00),v(84.40,26.29,0.00),v(84.54,26.15,0.00),v(84.64,26.01,0.00),v(84.71,25.89,0.00),v(84.73,25.77,0.00),v(84.72,25.67,0.00),v(84.68,25.59,0.00),v(84.61,25.51,0.00),v(84.51,25.44,0.00),v(84.23,25.34,0.00),v(83.86,25.28,0.00),v(83.41,25.26,0.00),v(82.89,25.29,0.00),v(82.33,25.35,0.00),v(81.74,25.45,0.00),v(81.13,25.59,0.00),v(80.52,25.77,0.00),v(79.92,25.98,0.00),v(79.35,26.22,0.00),v(78.83,26.50,0.00),v(78.36,26.82,0.00),v(78.16,26.98,0.00),v(77.90,27.22,0.00),v(77.58,27.53,0.00),v(77.22,27.89,0.00),v(76.82,28.32,0.00),v(76.39,28.79,0.00),v(75.92,29.31,0.00),v(75.44,29.87,0.00),v(74.93,30.46,0.00),v(74.42,31.09,0.00),v(73.39,32.42,0.00),v(72.39,33.82,0.00),v(71.48,35.24,0.00),v(70.23,36.06,0.00),v(69.01,36.80,0.00),v(67.81,37.44,0.00),v(66.65,38.00,0.00),v(66.09,38.25,0.00),v(65.55,38.47,0.00),v(65.03,38.67,0.00),v(64.53,38.85,0.00),v(64.05,39.00,0.00),v(63.59,39.13,0.00),v(63.15,39.23,0.00),v(62.75,39.31,0.00),v(62.37,39.37,0.00),v(62.02,39.40,0.00),v(61.71,39.40,0.00),v(61.43,39.38,0.00),v(61.18,39.34,0.00),v(60.98,39.27,0.00),v(60.81,39.17,0.00),v(60.68,39.05,0.00),v(60.60,38.90,0.00),v(60.56,38.72,0.00),v(60.57,38.52,0.00),v(60.62,38.29,0.00),v(60.73,38.03,0.00),v(60.89,37.74,0.00),v(61.10,37.43,0.00),v(61.37,37.08,0.00),v(61.70,36.69,0.00),v(62.05,36.29,0.00),v(62.81,35.45,0.00),v(63.64,34.57,0.00),v(64.50,33.67,0.00),v(65.37,32.77,0.00),v(66.23,31.87,0.00),v(67.05,31.00,0.00),v(67.82,30.17,0.00),v(68.29,29.62,0.00),v(68.68,29.09,0.00),v(68.99,28.59,0.00),v(69.22,28.12,0.00),v(69.36,27.68,0.00),v(69.40,27.26,0.00),v(69.35,26.89,0.00),v(69.20,26.54,0.00),v(68.95,26.23,0.00),v(68.59,25.96,0.00),v(68.13,25.73,0.00),v(67.85,25.62,0.00),v(67.55,25.53,0.00),v(67.22,25.45,0.00),v(66.85,25.38,0.00),v(66.46,25.32,0.00),v(66.04,25.27,0.00),v(65.59,25.23,0.00),v(65.11,25.20,0.00),v(64.59,25.18,0.00),v(64.04,25.17,0.00),v(63.31,25.18,0.00),v(62.62,25.18,0.00),v(61.97,25.20,0.00),v(61.35,25.21,0.00),v(60.77,25.24,0.00),v(60.21,25.27,0.00),v(59.20,25.37,0.00),v(58.29,25.50,0.00),v(57.47,25.68,0.00),v(56.72,25.90,0.00),v(56.03,26.19,0.00),v(55.38,26.53,0.00),v(54.75,26.95,0.00),v(54.13,27.44,0.00),v(53.51,28.00,0.00),v(52.87,28.66,0.00),v(52.18,29.40,0.00),v(51.82,29.81,0.00),v(51.44,30.24,0.00),v(51.05,30.70,0.00),v(50.63,31.18,0.00),v(50.49,31.38,0.00),v(50.37,31.61,0.00),v(50.29,31.86,0.00),v(50.23,32.12,0.00),v(50.18,32.66,0.00),v(50.22,33.17,0.00),v(50.27,33.40,0.00),v(50.34,33.59,0.00),v(50.43,33.76,0.00),v(50.54,33.87,0.00),v(50.66,33.94,0.00),v(50.79,33.95,0.00),v(50.94,33.90,0.00),v(51.10,33.78,0.00),v(51.59,33.29,0.00),v(52.05,32.84,0.00),v(52.49,32.42,0.00),v(52.90,32.02,0.00),v(53.29,31.66,0.00),v(53.65,31.33,0.00),v(54.00,31.02,0.00),v(54.33,30.74,0.00),v(54.96,30.25,0.00),v(55.55,29.85,0.00),v(56.12,29.54,0.00),v(56.69,29.29,0.00),v(57.28,29.11,0.00),v(57.92,28.99,0.00),v(58.61,28.90,0.00),v(59.38,28.86,0.00),v(60.24,28.84,0.00),v(61.22,28.83,0.00),v(62.34,28.84,0.00),v(62.95,28.84,0.00),v(63.60,28.84,0.00),v(63.13,29.31,0.00),v(62.85,29.55,0.00),v(62.56,29.81,0.00),v(62.24,30.09,0.00),v(61.89,30.39,0.00),v(61.52,30.72,0.00),v(61.12,31.08,0.00),v(60.70,31.48,0.00),v(60.24,31.93,0.00),v(59.76,32.42,0.00),v(59.25,32.96,0.00),v(58.71,33.56,0.00),v(58.13,34.21,0.00),v(57.53,34.94,0.00),v(56.89,35.73,0.00),v(56.36,36.47,0.00),v(55.96,37.18,0.00),v(55.66,37.86,0.00),v(55.48,38.52,0.00),v(55.40,39.15,0.00),v(55.42,39.74,0.00),v(55.53,40.29,0.00),v(55.74,40.80,0.00),v(56.03,41.26,0.00),v(56.41,41.67,0.00),v(56.87,42.03,0.00),v(57.41,42.33,0.00),v(58.02,42.57,0.00),v(58.69,42.75,0.00),v(59.43,42.86,0.00),v(60.23,42.89,0.00),v(60.71,42.88,0.00),v(61.21,42.84,0.00),v(61.73,42.77,0.00),v(62.27,42.68,0.00),v(62.84,42.56,0.00),v(63.42,42.41,0.00),v(64.01,42.23,0.00),v(64.62,42.03,0.00),v(65.87,41.54,0.00),v(67.14,40.93,0.00),v(68.41,40.22,0.00),v(69.68,39.39,0.00),v(69.61,39.77,0.00),v(69.58,40.14,0.00),v(69.59,40.49,0.00),v(69.62,40.82,0.00),v(69.69,41.13,0.00),v(69.80,41.42,0.00),v(69.95,41.69,0.00),v(70.13,41.94,0.00),v(70.36,42.16,0.00),v(70.63,42.36,0.00),v(70.95,42.53,0.00),v(71.31,42.67,0.00),v(71.72,42.78,0.00),v(72.18,42.86,0.00),v(72.69,42.91,0.00),v(73.25,42.93,0.00),v(73.94,42.91,0.00),v(74.64,42.83,0.00),v(75.35,42.72,0.00),v(76.07,42.55,0.00),v(76.79,42.35,0.00),v(77.52,42.12,0.00),v(78.97,41.54,0.00),v(80.41,40.86,0.00),v(81.82,40.08,0.00),v(83.20,39.24,0.00),v(84.53,38.35,0.00),v(84.47,38.83,0.00),v(84.45,39.29,0.00),v(84.48,39.74,0.00),v(84.55,40.17,0.00),v(84.67,40.57,0.00),v(84.83,40.95,0.00),v(85.05,41.30,0.00),v(85.31,41.62,0.00),v(85.62,41.91,0.00),v(85.98,42.17,0.00),v(86.40,42.40,0.00),v(86.87,42.58,0.00),v(87.40,42.73,0.00),v(87.98,42.84,0.00),v(88.61,42.91,0.00),v(89.31,42.93,0.00),v(89.87,42.92,0.00),v(90.45,42.88,0.00),v(91.03,42.80,0.00),v(91.63,42.70,0.00),v(92.23,42.57,0.00),v(92.84,42.40,0.00),v(93.46,42.20,0.00),v(94.09,41.96,0.00),v(94.72,41.69,0.00),v(95.37,41.38,0.00),v(96.01,41.03,0.00),v(96.67,40.64,0.00),v(97.32,40.21,0.00),v(97.98,39.74,0.00),v(98.65,39.22,0.00),v(99.32,38.66,0.00),v(99.34,39.10,0.00),v(99.39,39.53,0.00),v(99.49,39.94,0.00),v(99.62,40.33,0.00),v(99.79,40.70,0.00),v(100.00,41.05,0.00),v(100.24,41.38,0.00),v(100.53,41.68,0.00),v(100.85,41.95,0.00),v(101.20,42.19,0.00),v(101.60,42.40,0.00),v(102.02,42.58,0.00),v(102.49,42.72,0.00),v(102.98,42.83,0.00),v(103.52,42.89,0.00),v(104.08,42.91,0.00),v(104.45,42.90,0.00),v(104.85,42.88,0.00),v(105.71,42.80,0.00),v(106.68,42.68,0.00),v(107.73,42.51,0.00),v(108.86,42.30,0.00),v(110.08,42.06,0.00),v(111.36,41.80,0.00),v(112.70,41.51,0.00),v(114.10,41.19,0.00),v(115.55,40.87,0.00),v(117.05,40.53,0.00),v(118.58,40.18,0.00),v(121.73,39.50,0.00),v(124.95,38.84,0.00),v(128.18,38.24,0.00),v(129.79,37.98,0.00),v(131.39,37.74,0.00),v(132.96,37.55,0.00),v(134.50,37.39,0.00),v(136.01,37.27,0.00),v(137.48,37.20,0.00),v(138.90,37.18,0.00),v(140.26,37.22,0.00),v(141.57,37.32,0.00),v(142.81,37.48,0.00),v(143.97,37.72,0.00),v(144.52,37.86,0.00),v(145.05,38.03,0.00),v(145.56,38.21,0.00),v(146.05,38.42,0.00),v(146.51,38.64,0.00),v(146.95,38.89,0.00),v(146.96,38.88,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(91.36,39.51,0.00),v(91.14,39.50,0.00),v(90.94,39.48,0.00),v(90.60,39.38,0.00),v(90.34,39.23,0.00),v(90.16,39.02,0.00),v(90.05,38.76,0.00),v(90.02,38.46,0.00),v(90.05,38.12,0.00),v(90.14,37.73,0.00),v(90.30,37.32,0.00),v(90.51,36.87,0.00),v(90.78,36.40,0.00),v(91.09,35.91,0.00),v(91.46,35.41,0.00),v(91.86,34.89,0.00),v(92.31,34.36,0.00),v(92.80,33.83,0.00),v(93.86,32.77,0.00),v(95.04,31.74,0.00),v(95.66,31.25,0.00),v(96.29,30.78,0.00),v(96.94,30.34,0.00),v(97.60,29.92,0.00),v(98.26,29.54,0.00),v(98.93,29.20,0.00),v(99.60,28.89,0.00),v(100.27,28.64,0.00),v(100.93,28.43,0.00),v(101.57,28.28,0.00),v(102.21,28.18,0.00),v(102.82,28.15,0.00),v(103.15,28.18,0.00),v(103.39,28.28,0.00),v(103.55,28.43,0.00),v(103.64,28.64,0.00),v(103.66,28.90,0.00),v(103.61,29.20,0.00),v(103.51,29.54,0.00),v(103.34,29.93,0.00),v(103.12,30.34,0.00),v(102.84,30.79,0.00),v(102.52,31.26,0.00),v(102.16,31.75,0.00),v(101.75,32.25,0.00),v(101.31,32.77,0.00),v(100.34,33.83,0.00),v(99.26,34.89,0.00),v(98.11,35.92,0.00),v(96.91,36.88,0.00),v(95.71,37.74,0.00),v(95.11,38.12,0.00),v(94.52,38.46,0.00),v(93.94,38.77,0.00),v(93.38,39.02,0.00),v(92.83,39.23,0.00),v(92.31,39.38,0.00),v(91.81,39.48,0.00),v(91.35,39.51,0.00),v(91.36,39.51,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(87.41,21.94,0.00),v(88.22,21.26,0.00),v(88.94,20.62,0.00),v(89.58,20.03,0.00),v(90.14,19.49,0.00),v(90.63,18.98,0.00),v(91.04,18.52,0.00),v(91.38,18.10,0.00),v(91.66,17.71,0.00),v(91.88,17.36,0.00),v(92.04,17.05,0.00),v(92.15,16.77,0.00),v(92.20,16.53,0.00),v(92.21,16.31,0.00),v(92.18,16.13,0.00),v(92.11,15.97,0.00),v(92.00,15.84,0.00),v(91.86,15.74,0.00),v(91.70,15.66,0.00),v(91.51,15.61,0.00),v(91.30,15.58,0.00),v(90.83,15.57,0.00),v(90.33,15.64,0.00),v(89.83,15.76,0.00),v(89.35,15.94,0.00),v(88.92,16.15,0.00),v(88.58,16.39,0.00),v(88.26,16.66,0.00),v(87.91,16.97,0.00),v(87.16,17.63,0.00),v(86.37,18.36,0.00),v(85.55,19.13,0.00),v(84.73,19.92,0.00),v(83.95,20.71,0.00),v(83.23,21.47,0.00),v(82.90,21.84,0.00),v(82.59,22.19,0.00),v(82.47,22.38,0.00),v(82.45,22.53,0.00),v(82.52,22.67,0.00),v(82.68,22.77,0.00),v(82.90,22.85,0.00),v(83.18,22.90,0.00),v(83.52,22.92,0.00),v(83.90,22.92,0.00),v(84.32,22.89,0.00),v(84.76,22.83,0.00),v(85.68,22.64,0.00),v(86.14,22.50,0.00),v(86.59,22.34,0.00),v(87.01,22.16,0.00),v(87.40,21.95,0.00),v(87.41,21.94,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(48.18,34.04,0.00),v(47.79,34.41,0.00),v(47.35,34.81,0.00),v(46.86,35.22,0.00),v(46.33,35.64,0.00),v(45.76,36.06,0.00),v(45.16,36.48,0.00),v(44.53,36.90,0.00),v(43.88,37.29,0.00),v(43.20,37.67,0.00),v(42.51,38.02,0.00),v(41.81,38.33,0.00),v(41.10,38.60,0.00),v(40.38,38.82,0.00),v(39.66,38.98,0.00),v(38.95,39.09,0.00),v(38.25,39.12,0.00),v(38.01,39.11,0.00),v(37.82,39.07,0.00),v(37.68,39.00,0.00),v(37.58,38.90,0.00),v(37.52,38.78,0.00),v(37.49,38.64,0.00),v(37.51,38.47,0.00),v(37.55,38.29,0.00),v(37.63,38.08,0.00),v(37.74,37.86,0.00),v(37.88,37.61,0.00),v(38.04,37.35,0.00),v(38.44,36.78,0.00),v(38.92,36.16,0.00),v(39.46,35.49,0.00),v(40.05,34.78,0.00),v(41.31,33.28,0.00),v(42.57,31.73,0.00),v(43.16,30.96,0.00),v(43.70,30.21,0.00),v(43.93,29.86,0.00),v(44.13,29.52,0.00),v(44.31,29.21,0.00),v(44.46,28.90,0.00),v(44.58,28.61,0.00),v(44.69,28.34,0.00),v(44.83,27.83,0.00),v(44.89,27.38,0.00),v(44.87,26.97,0.00),v(44.79,26.62,0.00),v(44.64,26.31,0.00),v(44.43,26.05,0.00),v(44.18,25.83,0.00),v(43.89,25.65,0.00),v(43.56,25.50,0.00),v(43.20,25.40,0.00),v(42.82,25.32,0.00),v(42.03,25.27,0.00),v(41.04,25.31,0.00),v(40.05,25.43,0.00),v(39.07,25.63,0.00),v(38.10,25.89,0.00),v(37.14,26.21,0.00),v(36.20,26.58,0.00),v(35.27,27.00,0.00),v(34.36,27.46,0.00),v(33.47,27.94,0.00),v(32.61,28.45,0.00),v(30.96,29.51,0.00),v(30.18,30.05,0.00),v(29.43,30.58,0.00),v(28.71,31.11,0.00),v(28.03,31.61,0.00),v(28.46,31.06,0.00),v(28.93,30.47,0.00),v(29.45,29.84,0.00),v(30.01,29.17,0.00),v(30.60,28.45,0.00),v(31.21,27.70,0.00),v(31.85,26.89,0.00),v(32.51,26.04,0.00),v(32.58,25.93,0.00),v(32.61,25.82,0.00),v(32.61,25.73,0.00),v(32.57,25.64,0.00),v(32.51,25.57,0.00),v(32.41,25.50,0.00),v(32.13,25.40,0.00),v(31.76,25.34,0.00),v(31.32,25.32,0.00),v(30.81,25.33,0.00),v(30.25,25.39,0.00),v(29.67,25.48,0.00),v(29.07,25.61,0.00),v(28.48,25.78,0.00),v(27.90,25.99,0.00),v(27.37,26.23,0.00),v(26.88,26.51,0.00),v(26.47,26.82,0.00),v(26.14,27.17,0.00),v(25.37,28.17,0.00),v(24.55,29.26,0.00),v(23.57,30.38,0.00),v(22.54,31.45,0.00),v(21.47,32.47,0.00),v(20.36,33.43,0.00),v(19.23,34.33,0.00),v(18.09,35.17,0.00),v(16.95,35.95,0.00),v(15.81,36.65,0.00),v(14.69,37.29,0.00),v(13.59,37.85,0.00),v(12.52,38.33,0.00),v(11.50,38.74,0.00),v(10.54,39.05,0.00),v(10.08,39.18,0.00),v(9.63,39.29,0.00),v(9.21,39.37,0.00),v(8.80,39.43,0.00),v(8.42,39.46,0.00),v(8.05,39.47,0.00),v(7.67,39.46,0.00),v(7.33,39.41,0.00),v(7.02,39.32,0.00),v(6.74,39.21,0.00),v(6.50,39.06,0.00),v(6.29,38.87,0.00),v(6.13,38.66,0.00),v(6.00,38.41,0.00),v(5.91,38.13,0.00),v(5.86,37.82,0.00),v(5.86,37.48,0.00),v(5.89,37.10,0.00),v(5.98,36.70,0.00),v(6.10,36.26,0.00),v(6.28,35.80,0.00),v(6.50,35.30,0.00),v(6.81,34.67,0.00),v(7.18,34.00,0.00),v(7.58,33.30,0.00),v(8.02,32.58,0.00),v(8.49,31.83,0.00),v(9.00,31.06,0.00),v(9.54,30.27,0.00),v(10.11,29.46,0.00),v(10.70,28.63,0.00),v(11.32,27.79,0.00),v(12.62,26.08,0.00),v(13.99,24.34,0.00),v(15.41,22.60,0.00),v(16.87,20.87,0.00),v(18.33,19.18,0.00),v(19.80,17.53,0.00),v(21.24,15.95,0.00),v(22.63,14.47,0.00),v(23.31,13.76,0.00),v(23.97,13.09,0.00),v(24.61,12.44,0.00),v(25.23,11.83,0.00),v(25.82,11.26,0.00),v(26.38,10.72,0.00),v(26.60,10.50,0.00),v(26.77,10.30,0.00),v(26.87,10.11,0.00),v(26.92,9.94,0.00),v(26.92,9.80,0.00),v(26.87,9.67,0.00),v(26.78,9.56,0.00),v(26.65,9.46,0.00),v(26.48,9.39,0.00),v(26.27,9.33,0.00),v(26.03,9.28,0.00),v(25.77,9.25,0.00),v(25.16,9.23,0.00),v(24.48,9.27,0.00),v(23.74,9.36,0.00),v(22.97,9.50,0.00),v(22.21,9.68,0.00),v(21.46,9.90,0.00),v(20.76,10.16,0.00),v(20.13,10.44,0.00),v(19.59,10.75,0.00),v(19.37,10.92,0.00),v(19.18,11.09,0.00),v(16.12,14.22,0.00),v(13.14,17.39,0.00),v(11.69,18.99,0.00),v(10.28,20.59,0.00),v(8.93,22.18,0.00),v(7.63,23.77,0.00),v(6.41,25.34,0.00),v(5.26,26.90,0.00),v(4.20,28.43,0.00),v(3.23,29.94,0.00),v(2.37,31.41,0.00),v(1.97,32.14,0.00),v(1.61,32.86,0.00),v(1.28,33.56,0.00),v(0.98,34.26,0.00),v(0.71,34.95,0.00),v(0.48,35.62,0.00),v(0.27,36.36,0.00),v(0.12,37.08,0.00),v(0.03,37.78,0.00),v(0.00,38.46,0.00),v(0.04,39.11,0.00),v(0.14,39.73,0.00),v(0.32,40.31,0.00),v(0.56,40.85,0.00),v(0.88,41.34,0.00),v(1.27,41.78,0.00),v(1.74,42.17,0.00),v(2.28,42.50,0.00),v(2.91,42.76,0.00),v(3.25,42.86,0.00),v(3.61,42.95,0.00),v(4.00,43.02,0.00),v(4.40,43.07,0.00),v(4.83,43.10,0.00),v(5.28,43.11,0.00),v(6.04,43.09,0.00),v(6.80,43.02,0.00),v(7.58,42.91,0.00),v(8.36,42.76,0.00),v(9.93,42.36,0.00),v(11.50,41.82,0.00),v(13.06,41.17,0.00),v(14.60,40.43,0.00),v(16.10,39.62,0.00),v(17.55,38.75,0.00),v(16.86,39.68,0.00),v(16.22,40.56,0.00),v(15.62,41.37,0.00),v(15.08,42.10,0.00),v(15.00,42.23,0.00),v(14.96,42.35,0.00),v(14.94,42.46,0.00),v(14.96,42.56,0.00),v(15.02,42.64,0.00),v(15.09,42.72,0.00),v(15.33,42.84,0.00),v(15.65,42.92,0.00),v(16.05,42.96,0.00),v(16.50,42.96,0.00),v(17.00,42.93,0.00),v(17.52,42.86,0.00),v(18.05,42.76,0.00),v(18.58,42.63,0.00),v(19.08,42.47,0.00),v(19.55,42.28,0.00),v(19.97,42.07,0.00),v(20.31,41.83,0.00),v(20.57,41.57,0.00),v(20.99,41.05,0.00),v(21.44,40.53,0.00),v(22.44,39.47,0.00),v(23.53,38.40,0.00),v(24.71,37.34,0.00),v(25.94,36.30,0.00),v(27.22,35.29,0.00),v(28.52,34.32,0.00),v(29.83,33.40,0.00),v(31.12,32.55,0.00),v(32.37,31.77,0.00),v(32.98,31.41,0.00),v(33.57,31.08,0.00),v(34.14,30.77,0.00),v(34.69,30.49,0.00),v(35.21,30.23,0.00),v(35.71,30.01,0.00),v(36.18,29.81,0.00),v(36.62,29.65,0.00),v(37.03,29.52,0.00),v(37.40,29.43,0.00),v(37.73,29.37,0.00),v(38.02,29.35,0.00),v(38.23,29.37,0.00),v(38.38,29.43,0.00),v(38.46,29.52,0.00),v(38.49,29.65,0.00),v(38.47,29.81,0.00),v(38.40,30.00,0.00),v(38.28,30.22,0.00),v(38.12,30.47,0.00),v(37.93,30.74,0.00),v(37.70,31.04,0.00),v(37.44,31.36,0.00),v(37.15,31.70,0.00),v(36.51,32.43,0.00),v(35.82,33.23,0.00),v(35.09,34.08,0.00),v(34.37,34.96,0.00),v(33.67,35.86,0.00),v(33.04,36.77,0.00),v(32.50,37.67,0.00),v(32.27,38.12,0.00),v(32.08,38.56,0.00),v(31.92,38.98,0.00),v(31.81,39.40,0.00),v(31.74,39.81,0.00),v(31.72,40.21,0.00),v(31.77,40.69,0.00),v(31.88,41.12,0.00),v(32.06,41.50,0.00),v(32.31,41.84,0.00),v(32.61,42.13,0.00),v(32.96,42.37,0.00),v(33.36,42.57,0.00),v(33.81,42.73,0.00),v(34.30,42.85,0.00),v(34.83,42.92,0.00),v(35.40,42.96,0.00),v(35.99,42.95,0.00),v(36.62,42.91,0.00),v(37.27,42.84,0.00),v(37.93,42.73,0.00),v(38.62,42.59,0.00),v(39.31,42.41,0.00),v(40.01,42.21,0.00),v(41.43,41.71,0.00),v(42.84,41.10,0.00),v(43.53,40.75,0.00),v(44.20,40.39,0.00),v(44.86,40.00,0.00),v(45.49,39.58,0.00),v(46.10,39.15,0.00),v(46.68,38.70,0.00),v(47.22,38.23,0.00),v(47.73,37.75,0.00),v(48.20,37.25,0.00),v(48.62,36.73,0.00),v(48.73,36.55,0.00),v(48.82,36.34,0.00),v(48.88,36.10,0.00),v(48.91,35.85,0.00),v(48.93,35.34,0.00),v(48.87,34.84,0.00),v(48.75,34.40,0.00),v(48.67,34.23,0.00),v(48.59,34.09,0.00),v(48.49,34.00,0.00),v(48.39,33.95,0.00),v(48.28,33.97,0.00),v(48.17,34.04,0.00),v(48.18,34.04,0.00),})),}
unesco = merge{flip(linear_extrude_from_oriented(v(0,0,1),{v(28.65,42.90,0.00),v(51.42,42.90,0.00),v(51.42,41.94,0.00),v(28.65,41.94,0.00),v(28.65,42.90,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(27.38,44.53,0.00),v(52.66,44.53,0.00),v(52.66,43.58,0.00),v(27.38,43.58,0.00),v(27.38,44.53,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(26.19,46.14,0.00),v(53.90,46.14,0.00),v(53.90,45.18,0.00),v(26.19,45.18,0.00),v(26.19,46.14,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(28.65,30.62,0.00),v(51.42,30.62,0.00),v(51.42,29.66,0.00),v(28.65,29.66,0.00),v(28.65,30.62,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(40.05,26.13,0.00),v(30.51,29.04,0.00),v(33.27,29.04,0.00),v(40.05,27.01,0.00),v(46.68,29.04,0.00),v(49.45,29.04,0.00),v(40.05,26.13,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(40.05,24.50,0.00),v(28.65,28.02,0.00),v(28.65,29.04,0.00),v(39.96,25.60,0.00),v(51.42,29.04,0.00),v(51.42,28.02,0.00),v(40.05,24.50,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(35.39,41.26,0.00),v(35.39,31.30,0.00),v(34.83,31.30,0.00),v(34.82,33.97,0.00),v(34.52,33.97,0.00),v(34.21,31.30,0.00),v(33.64,31.30,0.00),v(33.64,41.26,0.00),v(34.21,41.26,0.00),v(34.21,38.27,0.00),v(34.52,38.27,0.00),v(34.83,41.26,0.00),v(35.39,41.26,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(38.21,36.32,0.00),v(38.83,36.32,0.00),v(38.83,35.67,0.00),v(38.21,35.67,0.00),v(38.21,31.92,0.00),v(38.83,31.92,0.00),v(38.83,31.30,0.00),v(37.54,31.30,0.00),v(37.54,41.26,0.00),v(39.06,41.26,0.00),v(39.06,40.63,0.00),v(38.21,40.63,0.00),v(38.21,36.32,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(31.67,40.56,0.00),v(31.67,31.30,0.00),v(31.07,31.30,0.00),v(31.07,40.58,0.00),v(31.07,40.61,0.00),v(31.05,40.67,0.00),v(30.99,40.73,0.00),v(30.88,40.75,0.00),v(30.71,40.75,0.00),v(30.59,40.73,0.00),v(30.53,40.67,0.00),v(30.51,40.61,0.00),v(30.51,40.58,0.00),v(30.51,31.30,0.00),v(29.92,31.30,0.00),v(29.92,40.56,0.00),v(29.91,40.59,0.00),v(29.92,40.67,0.00),v(29.94,40.78,0.00),v(29.99,40.91,0.00),v(30.07,41.04,0.00),v(30.20,41.15,0.00),v(30.39,41.23,0.00),v(30.65,41.26,0.00),v(30.65,41.26,0.00),v(30.96,41.26,0.00),v(30.96,41.26,0.00),v(31.20,41.22,0.00),v(31.38,41.14,0.00),v(31.51,41.03,0.00),v(31.59,40.90,0.00),v(31.64,40.77,0.00),v(31.66,40.66,0.00),v(31.67,40.58,0.00),v(31.67,40.56,0.00),v(31.67,40.56,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(42.70,40.56,0.00),v(42.70,40.48,0.00),v(42.70,40.29,0.00),v(42.70,40.00,0.00),v(42.70,39.66,0.00),v(42.70,38.95,0.00),v(42.70,38.64,0.00),v(42.70,38.41,0.00),v(42.69,38.18,0.00),v(42.65,37.93,0.00),v(42.52,37.36,0.00),v(42.33,36.75,0.00),v(42.11,36.12,0.00),v(41.89,35.49,0.00),v(41.70,34.90,0.00),v(41.62,34.63,0.00),v(41.57,34.38,0.00),v(41.53,34.15,0.00),v(41.51,33.95,0.00),v(41.51,32.00,0.00),v(41.51,31.97,0.00),v(41.52,31.89,0.00),v(41.56,31.81,0.00),v(41.66,31.78,0.00),v(41.66,31.78,0.00),v(41.91,31.78,0.00),v(41.91,31.78,0.00),v(42.01,31.81,0.00),v(42.06,31.89,0.00),v(42.08,31.97,0.00),v(42.08,32.00,0.00),v(42.08,34.23,0.00),v(42.70,34.23,0.00),v(42.70,32.00,0.00),v(42.70,31.97,0.00),v(42.69,31.89,0.00),v(42.67,31.78,0.00),v(42.62,31.65,0.00),v(42.53,31.52,0.00),v(42.40,31.41,0.00),v(42.22,31.33,0.00),v(41.97,31.30,0.00),v(41.97,31.30,0.00),v(41.94,31.30,0.00),v(41.94,31.30,0.00),v(41.94,31.30,0.00),v(41.68,31.30,0.00),v(41.68,31.30,0.00),v(41.68,31.30,0.00),v(41.66,31.30,0.00),v(41.66,31.30,0.00),v(41.40,31.34,0.00),v(41.22,31.42,0.00),v(41.08,31.53,0.00),v(40.99,31.66,0.00),v(40.93,31.79,0.00),v(40.91,31.90,0.00),v(40.90,31.97,0.00),v(40.89,32.00,0.00),v(40.89,33.92,0.00),v(40.91,34.17,0.00),v(40.94,34.43,0.00),v(41.08,34.99,0.00),v(41.27,35.58,0.00),v(41.49,36.19,0.00),v(41.70,36.78,0.00),v(41.89,37.34,0.00),v(42.03,37.83,0.00),v(42.07,38.05,0.00),v(42.08,38.24,0.00),v(42.08,40.56,0.00),v(42.08,40.59,0.00),v(42.06,40.65,0.00),v(42.01,40.71,0.00),v(41.91,40.72,0.00),v(41.91,40.72,0.00),v(41.66,40.72,0.00),v(41.66,40.72,0.00),v(41.56,40.70,0.00),v(41.51,40.64,0.00),v(41.49,40.58,0.00),v(41.49,40.56,0.00),v(41.49,38.04,0.00),v(40.87,38.04,0.00),v(40.87,40.56,0.00),v(40.87,40.59,0.00),v(40.87,40.67,0.00),v(40.90,40.78,0.00),v(40.95,40.91,0.00),v(41.03,41.04,0.00),v(41.16,41.15,0.00),v(41.35,41.23,0.00),v(41.60,41.26,0.00),v(41.60,41.26,0.00),v(41.63,41.26,0.00),v(41.63,41.26,0.00),v(41.63,41.26,0.00),v(41.88,41.26,0.00),v(41.88,41.26,0.00),v(41.88,41.26,0.00),v(41.91,41.26,0.00),v(41.91,41.26,0.00),v(42.17,41.23,0.00),v(42.37,41.15,0.00),v(42.51,41.04,0.00),v(42.60,40.91,0.00),v(42.66,40.78,0.00),v(42.69,40.67,0.00),v(42.70,40.59,0.00),v(42.70,40.56,0.00),v(42.70,40.56,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(46.43,34.26,0.00),v(46.43,32.03,0.00),v(46.42,32.00,0.00),v(46.41,31.92,0.00),v(46.38,31.81,0.00),v(46.32,31.68,0.00),v(46.22,31.55,0.00),v(46.08,31.44,0.00),v(45.89,31.36,0.00),v(45.64,31.33,0.00),v(45.64,31.33,0.00),v(45.38,31.33,0.00),v(45.38,31.33,0.00),v(45.38,31.33,0.00),v(45.38,31.33,0.00),v(45.38,31.33,0.00),v(45.14,31.36,0.00),v(44.95,31.44,0.00),v(44.82,31.55,0.00),v(44.73,31.68,0.00),v(44.67,31.81,0.00),v(44.64,31.92,0.00),v(44.62,32.00,0.00),v(44.62,32.03,0.00),v(44.62,36.18,0.00),v(44.62,36.27,0.00),v(44.62,40.56,0.00),v(44.62,40.59,0.00),v(44.63,40.67,0.00),v(44.66,40.78,0.00),v(44.71,40.91,0.00),v(44.81,41.04,0.00),v(44.94,41.15,0.00),v(45.13,41.23,0.00),v(45.38,41.26,0.00),v(45.38,41.26,0.00),v(45.38,41.26,0.00),v(45.38,41.26,0.00),v(45.38,41.26,0.00),v(45.64,41.26,0.00),v(45.64,41.26,0.00),v(45.89,41.23,0.00),v(46.08,41.15,0.00),v(46.22,41.04,0.00),v(46.32,40.91,0.00),v(46.38,40.78,0.00),v(46.41,40.67,0.00),v(46.42,40.59,0.00),v(46.43,40.56,0.00),v(46.43,38.33,0.00),v(45.83,38.33,0.00),v(45.83,40.56,0.00),v(45.83,40.59,0.00),v(45.81,40.67,0.00),v(45.75,40.75,0.00),v(45.64,40.78,0.00),v(45.38,40.78,0.00),v(45.25,40.75,0.00),v(45.18,40.67,0.00),v(45.16,40.59,0.00),v(45.16,40.56,0.00),v(45.16,36.27,0.00),v(45.16,36.18,0.00),v(45.16,32.03,0.00),v(45.16,32.00,0.00),v(45.18,31.92,0.00),v(45.25,31.84,0.00),v(45.38,31.81,0.00),v(45.64,31.81,0.00),v(45.75,31.84,0.00),v(45.81,31.92,0.00),v(45.83,32.00,0.00),v(45.83,32.03,0.00),v(45.83,34.26,0.00),v(46.43,34.26,0.00),v(46.43,34.26,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(49.39,31.30,0.00),v(49.30,31.30,0.00),v(49.22,31.30,0.00),v(48.95,31.33,0.00),v(48.75,31.41,0.00),v(48.60,31.52,0.00),v(48.50,31.65,0.00),v(48.44,31.78,0.00),v(48.41,31.89,0.00),v(48.40,31.97,0.00),v(48.40,32.00,0.00),v(48.40,36.15,0.00),v(48.40,36.24,0.00),v(48.40,40.53,0.00),v(48.40,40.56,0.00),v(48.41,40.64,0.00),v(48.44,40.75,0.00),v(48.50,40.88,0.00),v(48.60,41.01,0.00),v(48.75,41.12,0.00),v(48.95,41.20,0.00),v(49.22,41.23,0.00),v(49.30,41.23,0.00),v(49.39,41.23,0.00),v(49.66,41.20,0.00),v(49.86,41.12,0.00),v(50.01,41.01,0.00),v(50.11,40.88,0.00),v(50.16,40.75,0.00),v(50.19,40.64,0.00),v(50.21,40.56,0.00),v(50.21,40.53,0.00),v(50.21,36.24,0.00),v(50.21,36.15,0.00),v(50.21,32.00,0.00),v(50.21,31.97,0.00),v(50.19,31.89,0.00),v(50.16,31.78,0.00),v(50.11,31.65,0.00),v(50.01,31.52,0.00),v(49.86,31.41,0.00),v(49.66,31.33,0.00),v(49.39,31.30,0.00),v(49.39,31.30,0.00),})),flip(linear_extrude_from_oriented(v(0,0,1),{v(49.61,36.27,0.00),v(49.61,40.56,0.00),v(49.61,40.59,0.00),v(49.59,40.67,0.00),v(49.52,40.75,0.00),v(49.39,40.78,0.00),v(49.30,40.78,0.00),v(49.22,40.78,0.00),v(49.09,40.75,0.00),v(49.02,40.67,0.00),v(49.00,40.59,0.00),v(48.99,40.56,0.00),v(48.99,36.27,0.00),v(48.99,36.18,0.00),v(48.99,32.03,0.00),v(49.00,32.00,0.00),v(49.02,31.92,0.00),v(49.09,31.84,0.00),v(49.22,31.81,0.00),v(49.30,31.81,0.00),v(49.39,31.81,0.00),v(49.52,31.84,0.00),v(49.59,31.92,0.00),v(49.61,32.00,0.00),v(49.61,32.03,0.00),v(49.61,36.18,0.00),v(49.61,36.27,0.00),v(49.61,36.27,0.00),})),}
fh = translate(- v(148.76,214.26,0.00)) * merge{(linear_extrude_from_oriented(v(0,0,1),{v(148.76,214.26,0.00),v(148.76,215.37,0.00),v(148.42,215.37,0.00),v(148.42,212.73,0.00),v(149.83,212.73,0.00),v(149.83,213.05,0.00),v(148.76,213.05,0.00),v(148.76,213.94,0.00),v(149.72,213.94,0.00),v(149.72,214.26,0.00),v(148.76,214.26,0.00),v(148.76,214.26,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(150.48,215.37,0.00),v(150.16,215.37,0.00),v(150.16,213.45,0.00),v(150.47,213.45,0.00),v(150.47,213.76,0.00),v(150.66,213.60,0.00),v(150.83,213.49,0.00),v(151.00,213.43,0.00),v(151.17,213.41,0.00),v(151.17,213.73,0.00),v(151.00,213.76,0.00),v(150.83,213.84,0.00),v(150.66,213.97,0.00),v(150.48,214.13,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(152.47,215.11,0.00),v(152.29,215.25,0.00),v(152.12,215.34,0.00),v(151.96,215.39,0.00),v(151.81,215.41,0.00),v(151.69,215.40,0.00),v(151.59,215.37,0.00),v(151.44,215.25,0.00),v(151.35,215.07,0.00),v(151.32,214.85,0.00),v(151.35,214.63,0.00),v(151.45,214.45,0.00),v(151.54,214.39,0.00),v(151.65,214.34,0.00),v(151.79,214.31,0.00),v(151.95,214.30,0.00),v(152.18,214.31,0.00),v(152.43,214.34,0.00),v(152.43,214.08,0.00),v(152.41,213.90,0.00),v(152.34,213.79,0.00),v(152.22,213.72,0.00),v(152.06,213.71,0.00),v(151.79,213.74,0.00),v(151.50,213.84,0.00),v(151.40,213.58,0.00),v(151.74,213.45,0.00),v(152.07,213.41,0.00),v(152.26,213.42,0.00),v(152.41,213.46,0.00),v(152.52,213.52,0.00),v(152.61,213.60,0.00),v(152.71,213.81,0.00),v(152.74,214.06,0.00),v(152.74,214.89,0.00),v(152.78,215.04,0.00),v(152.83,215.08,0.00),v(152.92,215.10,0.00),v(152.96,215.10,0.00),v(152.96,215.39,0.00),v(152.90,215.39,0.00),v(152.74,215.37,0.00),v(152.61,215.31,0.00),v(152.52,215.22,0.00),v(152.47,215.11,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(151.94,215.05,0.00),v(152.03,215.04,0.00),v(152.14,214.98,0.00),v(152.27,214.88,0.00),v(152.44,214.73,0.00),v(152.43,214.65,0.00),v(152.19,214.60,0.00),v(151.97,214.58,0.00),v(151.82,214.59,0.00),v(151.71,214.64,0.00),v(151.65,214.72,0.00),v(151.63,214.84,0.00),v(151.65,214.95,0.00),v(151.70,215.02,0.00),v(151.78,215.06,0.00),v(151.89,215.06,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(153.63,213.99,0.00),v(153.63,215.37,0.00),v(153.31,215.37,0.00),v(153.31,213.45,0.00),v(153.62,213.45,0.00),v(153.62,213.72,0.00),v(153.81,213.57,0.00),v(153.99,213.48,0.00),v(154.16,213.43,0.00),v(154.31,213.41,0.00),v(154.55,213.45,0.00),v(154.72,213.58,0.00),v(154.83,213.78,0.00),v(154.87,214.06,0.00),v(154.87,215.37,0.00),v(154.55,215.37,0.00),v(154.55,214.11,0.00),v(154.53,213.93,0.00),v(154.47,213.81,0.00),v(154.37,213.73,0.00),v(154.24,213.71,0.00),v(154.12,213.73,0.00),v(153.98,213.80,0.00),v(153.82,213.91,0.00),v(153.63,214.06,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(156.07,216.27,0.00),v(155.88,216.26,0.00),v(155.71,216.23,0.00),v(155.74,215.96,0.00),v(155.88,215.99,0.00),v(156.01,216.00,0.00),v(156.09,215.99,0.00),v(156.15,215.97,0.00),v(156.19,215.89,0.00),v(156.15,215.84,0.00),v(156.02,215.80,0.00),v(155.96,215.79,0.00),v(155.91,215.80,0.00),v(155.82,215.63,0.00),v(155.85,215.39,0.00),v(155.72,215.35,0.00),v(155.60,215.29,0.00),v(155.43,215.10,0.00),v(155.33,214.85,0.00),v(155.29,214.52,0.00),v(155.29,214.30,0.00),v(155.31,214.10,0.00),v(155.34,213.93,0.00),v(155.40,213.78,0.00),v(155.49,213.65,0.00),v(155.59,213.55,0.00),v(155.73,213.47,0.00),v(155.88,213.43,0.00),v(156.05,213.41,0.00),v(156.34,213.45,0.00),v(156.64,213.55,0.00),v(156.55,213.82,0.00),v(156.30,213.74,0.00),v(156.07,213.71,0.00),v(155.87,213.74,0.00),v(155.73,213.86,0.00),v(155.64,214.04,0.00),v(155.62,214.30,0.00),v(155.62,214.52,0.00),v(155.64,214.78,0.00),v(155.73,214.97,0.00),v(155.87,215.08,0.00),v(156.07,215.12,0.00),v(156.31,215.09,0.00),v(156.57,215.00,0.00),v(156.67,215.27,0.00),v(156.40,215.37,0.00),v(156.13,215.41,0.00),v(156.10,215.56,0.00),v(156.26,215.58,0.00),v(156.38,215.65,0.00),v(156.46,215.75,0.00),v(156.48,215.88,0.00),v(156.46,216.04,0.00),v(156.39,216.16,0.00),v(156.27,216.24,0.00),v(156.09,216.26,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(157.72,213.70,0.00),v(157.52,213.74,0.00),v(157.36,213.86,0.00),v(157.27,214.04,0.00),v(157.24,214.29,0.00),v(157.24,214.52,0.00),v(157.27,214.78,0.00),v(157.36,214.97,0.00),v(157.52,215.08,0.00),v(157.72,215.12,0.00),v(157.93,215.08,0.00),v(158.08,214.97,0.00),v(158.18,214.78,0.00),v(158.21,214.52,0.00),v(158.21,214.29,0.00),v(158.18,214.04,0.00),v(158.08,213.86,0.00),v(157.93,213.74,0.00),v(157.72,213.70,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(158.54,214.52,0.00),v(158.52,214.72,0.00),v(158.48,214.90,0.00),v(158.42,215.05,0.00),v(158.33,215.17,0.00),v(158.21,215.28,0.00),v(158.07,215.35,0.00),v(157.91,215.39,0.00),v(157.72,215.41,0.00),v(157.54,215.39,0.00),v(157.38,215.35,0.00),v(157.24,215.28,0.00),v(157.12,215.17,0.00),v(157.03,215.05,0.00),v(156.96,214.90,0.00),v(156.92,214.72,0.00),v(156.91,214.52,0.00),v(156.91,214.29,0.00),v(156.92,214.10,0.00),v(156.96,213.93,0.00),v(157.03,213.78,0.00),v(157.12,213.65,0.00),v(157.23,213.55,0.00),v(157.37,213.47,0.00),v(157.54,213.43,0.00),v(157.72,213.41,0.00),v(157.91,213.43,0.00),v(158.07,213.47,0.00),v(158.21,213.55,0.00),v(158.33,213.65,0.00),v(158.42,213.78,0.00),v(158.48,213.93,0.00),v(158.52,214.10,0.00),v(158.54,214.29,0.00),v(158.54,214.52,0.00),v(158.54,214.52,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(159.02,213.45,0.00),v(159.34,213.45,0.00),v(159.34,215.37,0.00),v(159.02,215.37,0.00),v(159.02,213.45,0.00),v(159.02,213.45,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(159.00,212.62,0.00),v(159.36,212.62,0.00),v(159.36,212.98,0.00),v(159.00,212.98,0.00),v(159.00,212.62,0.00),v(159.00,212.62,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(160.79,214.86,0.00),v(160.78,214.76,0.00),v(160.72,214.68,0.00),v(160.62,214.62,0.00),v(160.47,214.57,0.00),v(160.25,214.52,0.00),v(160.05,214.44,0.00),v(159.90,214.31,0.00),v(159.81,214.16,0.00),v(159.78,213.97,0.00),v(159.82,213.75,0.00),v(159.94,213.58,0.00),v(160.03,213.51,0.00),v(160.14,213.46,0.00),v(160.27,213.42,0.00),v(160.43,213.41,0.00),v(160.73,213.44,0.00),v(161.05,213.55,0.00),v(160.95,213.82,0.00),v(160.67,213.73,0.00),v(160.43,213.69,0.00),v(160.29,213.71,0.00),v(160.18,213.76,0.00),v(160.12,213.83,0.00),v(160.10,213.95,0.00),v(160.11,214.04,0.00),v(160.16,214.12,0.00),v(160.25,214.18,0.00),v(160.39,214.23,0.00),v(160.60,214.28,0.00),v(160.83,214.36,0.00),v(160.98,214.48,0.00),v(161.07,214.64,0.00),v(161.10,214.84,0.00),v(161.06,215.07,0.00),v(160.95,215.25,0.00),v(160.85,215.32,0.00),v(160.73,215.37,0.00),v(160.59,215.40,0.00),v(160.41,215.41,0.00),v(160.08,215.38,0.00),v(159.74,215.26,0.00),v(159.83,215.00,0.00),v(160.13,215.10,0.00),v(160.41,215.13,0.00),v(160.58,215.11,0.00),v(160.70,215.06,0.00),v(160.77,214.98,0.00),v(160.79,214.86,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(149.23,218.05,0.00),v(148.24,218.05,0.00),v(148.24,219.16,0.00),v(147.78,219.16,0.00),v(147.78,216.52,0.00),v(148.24,216.52,0.00),v(148.24,217.60,0.00),v(149.23,217.60,0.00),v(149.23,216.52,0.00),v(149.70,216.52,0.00),v(149.70,219.16,0.00),v(149.23,219.16,0.00),v(149.23,218.05,0.00),v(149.23,218.05,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(150.95,217.55,0.00),v(150.78,217.58,0.00),v(150.65,217.68,0.00),v(150.57,217.84,0.00),v(150.54,218.05,0.00),v(150.54,218.29,0.00),v(150.57,218.50,0.00),v(150.65,218.66,0.00),v(150.78,218.76,0.00),v(150.95,218.79,0.00),v(151.12,218.76,0.00),v(151.25,218.66,0.00),v(151.33,218.50,0.00),v(151.35,218.29,0.00),v(151.35,218.05,0.00),v(151.33,217.84,0.00),v(151.25,217.68,0.00),v(151.12,217.58,0.00),v(150.95,217.55,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(151.81,218.29,0.00),v(151.80,218.48,0.00),v(151.75,218.66,0.00),v(151.68,218.82,0.00),v(151.58,218.95,0.00),v(151.46,219.05,0.00),v(151.31,219.13,0.00),v(151.14,219.18,0.00),v(150.95,219.19,0.00),v(150.75,219.18,0.00),v(150.58,219.13,0.00),v(150.43,219.05,0.00),v(150.31,218.95,0.00),v(150.21,218.82,0.00),v(150.14,218.66,0.00),v(150.10,218.48,0.00),v(150.09,218.29,0.00),v(150.09,218.05,0.00),v(150.10,217.86,0.00),v(150.14,217.68,0.00),v(150.21,217.53,0.00),v(150.31,217.40,0.00),v(150.43,217.29,0.00),v(150.58,217.21,0.00),v(150.75,217.16,0.00),v(150.95,217.15,0.00),v(151.14,217.16,0.00),v(151.32,217.21,0.00),v(151.46,217.29,0.00),v(151.59,217.40,0.00),v(151.68,217.53,0.00),v(151.75,217.68,0.00),v(151.80,217.86,0.00),v(151.81,218.05,0.00),v(151.81,218.29,0.00),v(151.81,218.29,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(152.18,216.29,0.00),v(152.63,216.29,0.00),v(152.63,218.61,0.00),v(152.66,218.72,0.00),v(152.70,218.75,0.00),v(152.77,218.77,0.00),v(152.83,218.77,0.00),v(152.83,219.18,0.00),v(152.72,219.18,0.00),v(152.57,219.17,0.00),v(152.45,219.13,0.00),v(152.36,219.07,0.00),v(152.29,219.00,0.00),v(152.20,218.81,0.00),v(152.18,218.61,0.00),v(152.18,216.29,0.00),v(152.18,216.29,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(153.16,216.29,0.00),v(153.61,216.29,0.00),v(153.61,218.61,0.00),v(153.64,218.72,0.00),v(153.68,218.75,0.00),v(153.75,218.77,0.00),v(153.81,218.77,0.00),v(153.81,219.18,0.00),v(153.70,219.18,0.00),v(153.55,219.17,0.00),v(153.43,219.13,0.00),v(153.34,219.07,0.00),v(153.27,219.00,0.00),v(153.18,218.81,0.00),v(153.16,218.61,0.00),v(153.16,216.29,0.00),v(153.16,216.29,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(154.70,218.83,0.00),v(154.80,218.82,0.00),v(154.91,218.78,0.00),v(155.04,218.70,0.00),v(155.20,218.56,0.00),v(155.20,218.40,0.00),v(154.80,218.36,0.00),v(154.67,218.38,0.00),v(154.59,218.42,0.00),v(154.54,218.50,0.00),v(154.52,218.60,0.00),v(154.53,218.70,0.00),v(154.57,218.77,0.00),v(154.63,218.81,0.00),v(154.70,218.83,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(155.25,218.90,0.00),v(155.09,219.03,0.00),v(154.93,219.13,0.00),v(154.77,219.18,0.00),v(154.60,219.20,0.00),v(154.47,219.19,0.00),v(154.37,219.16,0.00),v(154.21,219.03,0.00),v(154.12,218.85,0.00),v(154.09,218.61,0.00),v(154.13,218.38,0.00),v(154.24,218.19,0.00),v(154.33,218.12,0.00),v(154.45,218.07,0.00),v(154.59,218.04,0.00),v(154.77,218.03,0.00),v(155.20,218.08,0.00),v(155.20,217.85,0.00),v(155.17,217.71,0.00),v(155.11,217.63,0.00),v(155.00,217.58,0.00),v(154.87,217.57,0.00),v(154.61,217.60,0.00),v(154.32,217.69,0.00),v(154.19,217.31,0.00),v(154.56,217.19,0.00),v(154.90,217.15,0.00),v(155.09,217.17,0.00),v(155.25,217.21,0.00),v(155.38,217.28,0.00),v(155.48,217.37,0.00),v(155.59,217.59,0.00),v(155.62,217.85,0.00),v(155.62,218.61,0.00),v(155.66,218.72,0.00),v(155.71,218.75,0.00),v(155.79,218.77,0.00),v(155.83,218.77,0.00),v(155.83,219.18,0.00),v(155.74,219.18,0.00),v(155.56,219.16,0.00),v(155.42,219.10,0.00),v(155.32,219.01,0.00),v(155.25,218.90,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(156.57,217.80,0.00),v(156.57,219.16,0.00),v(156.13,219.16,0.00),v(156.13,217.19,0.00),v(156.55,217.19,0.00),v(156.55,217.46,0.00),v(156.73,217.32,0.00),v(156.89,217.23,0.00),v(157.06,217.17,0.00),v(157.22,217.15,0.00),v(157.36,217.17,0.00),v(157.47,217.20,0.00),v(157.57,217.26,0.00),v(157.65,217.33,0.00),v(157.72,217.43,0.00),v(157.77,217.55,0.00),v(157.80,217.84,0.00),v(157.80,219.16,0.00),v(157.36,219.16,0.00),v(157.36,217.90,0.00),v(157.34,217.75,0.00),v(157.29,217.65,0.00),v(157.21,217.59,0.00),v(157.09,217.57,0.00),v(156.98,217.58,0.00),v(156.87,217.62,0.00),v(156.73,217.69,0.00),v(156.57,217.80,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(158.97,218.78,0.00),v(159.16,218.73,0.00),v(159.27,218.66,0.00),v(159.41,218.53,0.00),v(159.41,217.63,0.00),v(159.22,217.57,0.00),v(159.06,217.55,0.00),v(158.87,217.59,0.00),v(158.74,217.69,0.00),v(158.66,217.86,0.00),v(158.64,218.07,0.00),v(158.64,218.28,0.00),v(158.66,218.51,0.00),v(158.72,218.66,0.00),v(158.83,218.76,0.00),v(158.97,218.78,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(159.43,218.89,0.00),v(159.26,219.04,0.00),v(159.12,219.14,0.00),v(158.99,219.19,0.00),v(158.85,219.20,0.00),v(158.70,219.18,0.00),v(158.57,219.14,0.00),v(158.45,219.06,0.00),v(158.35,218.96,0.00),v(158.28,218.83,0.00),v(158.22,218.67,0.00),v(158.19,218.49,0.00),v(158.18,218.28,0.00),v(158.18,218.07,0.00),v(158.19,217.87,0.00),v(158.23,217.69,0.00),v(158.30,217.54,0.00),v(158.39,217.40,0.00),v(158.51,217.30,0.00),v(158.65,217.22,0.00),v(158.82,217.17,0.00),v(159.01,217.15,0.00),v(159.41,217.23,0.00),v(159.41,216.29,0.00),v(159.85,216.29,0.00),v(159.85,219.16,0.00),v(159.43,219.16,0.00),v(159.43,218.89,0.00),v(159.43,218.89,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(160.67,217.97,0.00),v(161.42,217.97,0.00),v(161.42,217.90,0.00),v(161.39,217.74,0.00),v(161.32,217.62,0.00),v(161.20,217.55,0.00),v(161.06,217.53,0.00),v(160.91,217.56,0.00),v(160.79,217.65,0.00),v(160.71,217.78,0.00),v(160.67,217.97,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(161.13,219.20,0.00),v(160.92,219.18,0.00),v(160.74,219.14,0.00),v(160.59,219.06,0.00),v(160.46,218.95,0.00),v(160.36,218.82,0.00),v(160.29,218.67,0.00),v(160.25,218.49,0.00),v(160.24,218.29,0.00),v(160.24,218.06,0.00),v(160.25,217.86,0.00),v(160.29,217.69,0.00),v(160.36,217.53,0.00),v(160.46,217.40,0.00),v(160.58,217.30,0.00),v(160.72,217.22,0.00),v(160.88,217.17,0.00),v(161.06,217.15,0.00),v(161.36,217.20,0.00),v(161.49,217.26,0.00),v(161.61,217.35,0.00),v(161.72,217.46,0.00),v(161.79,217.59,0.00),v(161.84,217.76,0.00),v(161.86,217.95,0.00),v(161.85,218.12,0.00),v(161.81,218.32,0.00),v(160.67,218.32,0.00),v(160.70,218.53,0.00),v(160.79,218.68,0.00),v(160.94,218.77,0.00),v(161.15,218.80,0.00),v(161.42,218.77,0.00),v(161.72,218.67,0.00),v(161.84,219.05,0.00),v(161.46,219.17,0.00),v(161.13,219.20,0.00),})),}
ib = translate(- v(68.28,128.92,0.00)) * merge{(linear_extrude_from_oriented(v(0,0,1),{v(68.28,128.92,0.00),v(68.62,128.92,0.00),v(68.62,131.56,0.00),v(68.28,131.56,0.00),v(68.28,128.92,0.00),v(68.28,128.92,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(69.60,131.56,0.00),v(69.28,131.56,0.00),v(69.28,129.64,0.00),v(69.59,129.64,0.00),v(69.59,129.95,0.00),v(69.78,129.79,0.00),v(69.95,129.68,0.00),v(70.12,129.62,0.00),v(70.29,129.60,0.00),v(70.29,129.92,0.00),v(70.12,129.95,0.00),v(69.96,130.03,0.00),v(69.79,130.15,0.00),v(69.61,130.31,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(70.62,129.64,0.00),v(70.93,129.64,0.00),v(70.93,131.56,0.00),v(70.62,131.56,0.00),v(70.62,129.64,0.00),v(70.62,129.64,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(70.60,128.81,0.00),v(70.95,128.81,0.00),v(70.95,129.17,0.00),v(70.60,129.17,0.00),v(70.60,128.81,0.00),v(70.60,128.81,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(71.85,130.18,0.00),v(71.85,131.56,0.00),v(71.53,131.56,0.00),v(71.53,129.64,0.00),v(71.83,129.64,0.00),v(71.83,129.91,0.00),v(72.03,129.76,0.00),v(72.21,129.67,0.00),v(72.37,129.62,0.00),v(72.53,129.60,0.00),v(72.77,129.64,0.00),v(72.94,129.77,0.00),v(73.05,129.97,0.00),v(73.08,130.25,0.00),v(73.08,131.56,0.00),v(72.76,131.56,0.00),v(72.76,130.30,0.00),v(72.74,130.12,0.00),v(72.69,130.00,0.00),v(72.59,129.92,0.00),v(72.45,129.90,0.00),v(72.33,129.92,0.00),v(72.20,130.00,0.00),v(72.04,130.11,0.00),v(71.85,130.26,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(74.62,131.30,0.00),v(74.44,131.44,0.00),v(74.28,131.53,0.00),v(74.12,131.58,0.00),v(73.97,131.60,0.00),v(73.85,131.59,0.00),v(73.75,131.56,0.00),v(73.59,131.44,0.00),v(73.50,131.26,0.00),v(73.47,131.04,0.00),v(73.51,130.82,0.00),v(73.61,130.64,0.00),v(73.70,130.58,0.00),v(73.81,130.53,0.00),v(73.94,130.50,0.00),v(74.11,130.49,0.00),v(74.34,130.50,0.00),v(74.59,130.53,0.00),v(74.59,130.27,0.00),v(74.56,130.09,0.00),v(74.49,129.98,0.00),v(74.38,129.91,0.00),v(74.21,129.90,0.00),v(73.95,129.93,0.00),v(73.66,130.03,0.00),v(73.56,129.77,0.00),v(73.90,129.64,0.00),v(74.23,129.60,0.00),v(74.41,129.61,0.00),v(74.56,129.65,0.00),v(74.68,129.71,0.00),v(74.77,129.79,0.00),v(74.87,130.00,0.00),v(74.90,130.25,0.00),v(74.90,131.08,0.00),v(74.93,131.23,0.00),v(74.99,131.27,0.00),v(75.07,131.29,0.00),v(75.12,131.29,0.00),v(75.12,131.58,0.00),v(75.06,131.58,0.00),v(74.90,131.56,0.00),v(74.77,131.50,0.00),v(74.68,131.41,0.00),v(74.62,131.30,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(74.03,131.33,0.00),v(74.14,131.30,0.00),v(74.26,131.21,0.00),v(74.41,131.09,0.00),v(74.59,130.92,0.00),v(74.59,130.82,0.00),v(74.34,130.78,0.00),v(74.12,130.77,0.00),v(73.97,130.78,0.00),v(73.86,130.83,0.00),v(73.81,130.91,0.00),v(73.79,131.03,0.00),v(73.80,131.15,0.00),v(73.85,131.25,0.00),v(73.93,131.31,0.00),v(74.03,131.33,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(66.53,134.18,0.00),v(66.53,134.87,0.00),v(66.96,134.87,0.00),v(67.10,134.85,0.00),v(67.21,134.80,0.00),v(67.27,134.71,0.00),v(67.29,134.59,0.00),v(67.29,134.46,0.00),v(67.27,134.34,0.00),v(67.21,134.25,0.00),v(67.10,134.19,0.00),v(66.96,134.18,0.00),v(66.53,134.18,0.00),v(66.53,134.18,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(66.53,133.09,0.00),v(66.53,133.75,0.00),v(66.86,133.75,0.00),v(67.00,133.73,0.00),v(67.10,133.68,0.00),v(67.16,133.59,0.00),v(67.18,133.48,0.00),v(67.18,133.35,0.00),v(67.16,133.24,0.00),v(67.10,133.16,0.00),v(67.00,133.11,0.00),v(66.86,133.09,0.00),v(66.53,133.09,0.00),v(66.53,133.09,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(66.06,132.66,0.00),v(66.86,132.66,0.00),v(67.07,132.67,0.00),v(67.23,132.71,0.00),v(67.37,132.77,0.00),v(67.48,132.85,0.00),v(67.56,132.95,0.00),v(67.61,133.07,0.00),v(67.66,133.35,0.00),v(67.66,133.46,0.00),v(67.64,133.61,0.00),v(67.59,133.73,0.00),v(67.50,133.84,0.00),v(67.39,133.93,0.00),v(67.56,134.02,0.00),v(67.67,134.14,0.00),v(67.75,134.30,0.00),v(67.77,134.48,0.00),v(67.77,134.59,0.00),v(67.76,134.74,0.00),v(67.73,134.88,0.00),v(67.67,135.00,0.00),v(67.59,135.11,0.00),v(67.48,135.19,0.00),v(67.34,135.25,0.00),v(67.17,135.29,0.00),v(66.96,135.30,0.00),v(66.06,135.30,0.00),v(66.06,132.66,0.00),v(66.06,132.66,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(68.93,133.69,0.00),v(68.76,133.73,0.00),v(68.63,133.83,0.00),v(68.55,133.99,0.00),v(68.53,134.20,0.00),v(68.53,134.43,0.00),v(68.55,134.64,0.00),v(68.63,134.80,0.00),v(68.76,134.90,0.00),v(68.93,134.94,0.00),v(69.10,134.90,0.00),v(69.23,134.80,0.00),v(69.31,134.64,0.00),v(69.33,134.43,0.00),v(69.33,134.20,0.00),v(69.31,133.99,0.00),v(69.23,133.83,0.00),v(69.10,133.73,0.00),v(68.93,133.69,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(69.79,134.43,0.00),v(69.78,134.63,0.00),v(69.73,134.81,0.00),v(69.66,134.96,0.00),v(69.57,135.09,0.00),v(69.44,135.20,0.00),v(69.29,135.28,0.00),v(69.12,135.32,0.00),v(68.93,135.34,0.00),v(68.73,135.32,0.00),v(68.56,135.28,0.00),v(68.41,135.20,0.00),v(68.29,135.09,0.00),v(68.19,134.96,0.00),v(68.12,134.81,0.00),v(68.08,134.63,0.00),v(68.07,134.43,0.00),v(68.07,134.20,0.00),v(68.08,134.00,0.00),v(68.12,133.83,0.00),v(68.19,133.67,0.00),v(68.29,133.54,0.00),v(68.41,133.44,0.00),v(68.56,133.36,0.00),v(68.73,133.31,0.00),v(68.93,133.29,0.00),v(69.12,133.31,0.00),v(69.30,133.36,0.00),v(69.44,133.44,0.00),v(69.57,133.54,0.00),v(69.66,133.67,0.00),v(69.73,133.83,0.00),v(69.78,134.00,0.00),v(69.79,134.20,0.00),v(69.79,134.43,0.00),v(69.79,134.43,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(70.89,134.48,0.00),v(70.65,134.71,0.00),v(70.65,135.30,0.00),v(70.20,135.30,0.00),v(70.20,132.43,0.00),v(70.65,132.43,0.00),v(70.65,134.22,0.00),v(70.85,134.00,0.00),v(71.03,133.79,0.00),v(71.32,133.33,0.00),v(71.84,133.33,0.00),v(71.56,133.78,0.00),v(71.22,134.20,0.00),v(71.91,135.30,0.00),v(71.37,135.30,0.00),v(70.89,134.48,0.00),v(70.89,134.48,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(72.84,133.69,0.00),v(72.67,133.73,0.00),v(72.54,133.83,0.00),v(72.46,133.99,0.00),v(72.43,134.20,0.00),v(72.43,134.43,0.00),v(72.46,134.64,0.00),v(72.54,134.80,0.00),v(72.67,134.90,0.00),v(72.84,134.94,0.00),v(73.01,134.90,0.00),v(73.14,134.80,0.00),v(73.22,134.64,0.00),v(73.24,134.43,0.00),v(73.24,134.20,0.00),v(73.22,133.99,0.00),v(73.14,133.83,0.00),v(73.01,133.73,0.00),v(72.84,133.69,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(73.70,134.43,0.00),v(73.69,134.63,0.00),v(73.64,134.81,0.00),v(73.57,134.96,0.00),v(73.47,135.09,0.00),v(73.35,135.20,0.00),v(73.20,135.28,0.00),v(73.03,135.32,0.00),v(72.84,135.34,0.00),v(72.64,135.32,0.00),v(72.47,135.28,0.00),v(72.32,135.20,0.00),v(72.20,135.09,0.00),v(72.10,134.96,0.00),v(72.03,134.81,0.00),v(71.99,134.63,0.00),v(71.98,134.43,0.00),v(71.98,134.20,0.00),v(71.99,134.00,0.00),v(72.03,133.83,0.00),v(72.10,133.67,0.00),v(72.20,133.54,0.00),v(72.32,133.44,0.00),v(72.47,133.36,0.00),v(72.64,133.31,0.00),v(72.84,133.29,0.00),v(73.03,133.31,0.00),v(73.21,133.36,0.00),v(73.35,133.44,0.00),v(73.48,133.54,0.00),v(73.57,133.67,0.00),v(73.64,133.83,0.00),v(73.69,134.00,0.00),v(73.70,134.20,0.00),v(73.70,134.43,0.00),v(73.70,134.43,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(74.92,135.30,0.00),v(74.51,135.30,0.00),v(73.81,133.33,0.00),v(74.26,133.33,0.00),v(74.60,134.32,0.00),v(74.73,134.72,0.00),v(74.85,134.32,0.00),v(75.17,133.33,0.00),v(75.61,133.33,0.00),v(74.92,135.30,0.00),v(74.92,135.30,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(76.34,134.97,0.00),v(76.44,134.96,0.00),v(76.55,134.92,0.00),v(76.68,134.84,0.00),v(76.84,134.70,0.00),v(76.84,134.54,0.00),v(76.44,134.50,0.00),v(76.31,134.52,0.00),v(76.23,134.56,0.00),v(76.18,134.64,0.00),v(76.16,134.74,0.00),v(76.17,134.84,0.00),v(76.21,134.91,0.00),v(76.27,134.95,0.00),v(76.34,134.97,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(76.89,135.04,0.00),v(76.73,135.17,0.00),v(76.57,135.27,0.00),v(76.41,135.32,0.00),v(76.24,135.34,0.00),v(76.11,135.33,0.00),v(76.01,135.30,0.00),v(75.85,135.17,0.00),v(75.76,134.99,0.00),v(75.73,134.75,0.00),v(75.77,134.52,0.00),v(75.88,134.33,0.00),v(75.97,134.26,0.00),v(76.09,134.21,0.00),v(76.23,134.18,0.00),v(76.41,134.17,0.00),v(76.84,134.22,0.00),v(76.84,133.99,0.00),v(76.81,133.85,0.00),v(76.75,133.77,0.00),v(76.64,133.72,0.00),v(76.51,133.71,0.00),v(76.25,133.74,0.00),v(75.96,133.83,0.00),v(75.83,133.45,0.00),v(76.20,133.33,0.00),v(76.54,133.29,0.00),v(76.73,133.31,0.00),v(76.89,133.35,0.00),v(77.02,133.42,0.00),v(77.12,133.50,0.00),v(77.23,133.73,0.00),v(77.26,133.99,0.00),v(77.26,134.75,0.00),v(77.30,134.86,0.00),v(77.35,134.89,0.00),v(77.43,134.91,0.00),v(77.47,134.91,0.00),v(77.47,135.32,0.00),v(77.38,135.32,0.00),v(77.20,135.30,0.00),v(77.06,135.24,0.00),v(76.96,135.15,0.00),v(76.89,135.04,0.00),})),}
so = translate(- v(21.04,22.22,0.00)) * merge{(linear_extrude_from_oriented(v(0,0,1),{v(21.04,22.22,0.00),v(21.03,22.14,0.00),v(20.99,22.08,0.00),v(20.91,22.03,0.00),v(20.79,21.99,0.00),v(20.59,21.93,0.00),v(20.39,21.85,0.00),v(20.24,21.73,0.00),v(20.14,21.56,0.00),v(20.11,21.37,0.00),v(20.15,21.15,0.00),v(20.28,20.97,0.00),v(20.38,20.90,0.00),v(20.49,20.85,0.00),v(20.63,20.82,0.00),v(20.79,20.80,0.00),v(21.09,20.83,0.00),v(21.41,20.95,0.00),v(21.27,21.31,0.00),v(20.99,21.20,0.00),v(20.76,21.17,0.00),v(20.65,21.18,0.00),v(20.57,21.21,0.00),v(20.52,21.27,0.00),v(20.51,21.34,0.00),v(20.56,21.45,0.00),v(20.63,21.50,0.00),v(20.73,21.54,0.00),v(20.93,21.60,0.00),v(21.16,21.68,0.00),v(21.32,21.80,0.00),v(21.41,21.96,0.00),v(21.44,22.16,0.00),v(21.40,22.40,0.00),v(21.26,22.58,0.00),v(21.16,22.65,0.00),v(21.03,22.71,0.00),v(20.88,22.74,0.00),v(20.70,22.75,0.00),v(20.38,22.71,0.00),v(20.03,22.61,0.00),v(20.14,22.27,0.00),v(20.45,22.37,0.00),v(20.70,22.41,0.00),v(20.84,22.39,0.00),v(20.95,22.35,0.00),v(21.02,22.29,0.00),v(21.04,22.22,0.00),v(21.04,22.22,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(23.39,21.91,0.00),v(23.37,22.09,0.00),v(23.33,22.25,0.00),v(23.26,22.40,0.00),v(23.17,22.52,0.00),v(23.05,22.62,0.00),v(22.91,22.69,0.00),v(22.75,22.74,0.00),v(22.57,22.75,0.00),v(22.39,22.74,0.00),v(22.23,22.69,0.00),v(22.09,22.62,0.00),v(21.97,22.52,0.00),v(21.87,22.40,0.00),v(21.81,22.25,0.00),v(21.76,22.09,0.00),v(21.75,21.91,0.00),v(21.75,21.68,0.00),v(21.76,21.50,0.00),v(21.80,21.33,0.00),v(21.87,21.19,0.00),v(21.96,21.07,0.00),v(22.07,20.97,0.00),v(22.21,20.89,0.00),v(22.38,20.85,0.00),v(22.57,20.83,0.00),v(22.75,20.85,0.00),v(22.91,20.89,0.00),v(23.05,20.97,0.00),v(23.17,21.07,0.00),v(23.26,21.19,0.00),v(23.33,21.33,0.00),v(23.37,21.50,0.00),v(23.39,21.68,0.00),v(23.39,21.91,0.00),v(23.39,21.91,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(22.57,21.20,0.00),v(22.40,21.23,0.00),v(22.27,21.32,0.00),v(22.20,21.47,0.00),v(22.17,21.68,0.00),v(22.17,21.91,0.00),v(22.20,22.11,0.00),v(22.27,22.26,0.00),v(22.40,22.35,0.00),v(22.57,22.39,0.00),v(22.74,22.35,0.00),v(22.86,22.26,0.00),v(22.94,22.11,0.00),v(22.96,21.91,0.00),v(22.96,21.68,0.00),v(22.94,21.49,0.00),v(22.86,21.33,0.00),v(22.74,21.24,0.00),v(22.57,21.20,0.00),v(22.57,21.20,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(24.97,22.47,0.00),v(24.80,22.60,0.00),v(24.65,22.69,0.00),v(24.35,22.75,0.00),v(24.12,22.71,0.00),v(23.96,22.59,0.00),v(23.87,22.38,0.00),v(23.84,22.10,0.00),v(23.84,20.86,0.00),v(24.26,20.86,0.00),v(24.26,22.05,0.00),v(24.28,22.18,0.00),v(24.32,22.28,0.00),v(24.39,22.34,0.00),v(24.49,22.36,0.00),v(24.58,22.35,0.00),v(24.68,22.31,0.00),v(24.80,22.24,0.00),v(24.94,22.13,0.00),v(24.94,20.83,0.00),v(25.36,20.83,0.00),v(25.36,22.72,0.00),v(24.97,22.72,0.00),v(24.97,22.47,0.00),v(24.97,22.47,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(26.58,22.75,0.00),v(26.42,22.74,0.00),v(26.29,22.70,0.00),v(26.19,22.64,0.00),v(26.11,22.57,0.00),v(26.01,22.36,0.00),v(25.98,22.10,0.00),v(25.98,21.23,0.00),v(25.67,21.23,0.00),v(25.67,20.86,0.00),v(25.98,20.86,0.00),v(25.98,20.38,0.00),v(26.41,20.38,0.00),v(26.41,20.86,0.00),v(27.00,20.86,0.00),v(27.00,21.23,0.00),v(26.41,21.23,0.00),v(26.41,22.10,0.00),v(26.42,22.23,0.00),v(26.47,22.30,0.00),v(26.55,22.35,0.00),v(26.66,22.36,0.00),v(26.79,22.34,0.00),v(26.94,22.30,0.00),v(27.05,22.67,0.00),v(26.83,22.73,0.00),v(26.58,22.75,0.00),v(26.58,22.75,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(27.39,20.52,0.00),v(27.39,20.10,0.00),v(27.85,20.10,0.00),v(27.85,20.52,0.00),v(27.39,20.52,0.00),v(27.39,20.52,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(27.39,22.72,0.00),v(27.39,20.86,0.00),v(27.82,20.86,0.00),v(27.82,22.72,0.00),v(27.39,22.72,0.00),v(27.39,22.72,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(29.12,22.75,0.00),v(28.92,22.74,0.00),v(28.74,22.69,0.00),v(28.60,22.62,0.00),v(28.48,22.52,0.00),v(28.39,22.40,0.00),v(28.32,22.25,0.00),v(28.28,22.09,0.00),v(28.27,21.91,0.00),v(28.27,21.68,0.00),v(28.28,21.50,0.00),v(28.32,21.33,0.00),v(28.39,21.19,0.00),v(28.48,21.07,0.00),v(28.60,20.97,0.00),v(28.73,20.89,0.00),v(28.89,20.85,0.00),v(29.06,20.83,0.00),v(29.34,20.88,0.00),v(29.47,20.93,0.00),v(29.59,21.01,0.00),v(29.68,21.12,0.00),v(29.76,21.25,0.00),v(29.80,21.41,0.00),v(29.82,21.60,0.00),v(29.76,21.93,0.00),v(28.69,21.93,0.00),v(28.72,22.14,0.00),v(28.80,22.28,0.00),v(28.94,22.36,0.00),v(29.14,22.39,0.00),v(29.40,22.36,0.00),v(29.68,22.27,0.00),v(29.79,22.64,0.00),v(29.44,22.73,0.00),v(29.12,22.75,0.00),v(29.12,22.75,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(28.69,21.60,0.00),v(29.40,21.60,0.00),v(29.40,21.54,0.00),v(29.37,21.39,0.00),v(29.30,21.28,0.00),v(29.19,21.22,0.00),v(29.06,21.20,0.00),v(28.91,21.21,0.00),v(28.80,21.29,0.00),v(28.73,21.42,0.00),v(28.69,21.60,0.00),v(28.69,21.60,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(30.70,21.62,0.00),v(30.70,22.72,0.00),v(30.27,22.72,0.00),v(30.27,20.86,0.00),v(30.67,20.86,0.00),v(30.67,21.12,0.00),v(30.83,20.99,0.00),v(30.99,20.90,0.00),v(31.15,20.85,0.00),v(31.32,20.83,0.00),v(31.44,20.84,0.00),v(31.55,20.88,0.00),v(31.65,20.93,0.00),v(31.72,21.00,0.00),v(31.82,21.20,0.00),v(31.85,21.48,0.00),v(31.85,22.72,0.00),v(31.43,22.72,0.00),v(31.43,21.54,0.00),v(31.41,21.40,0.00),v(31.35,21.30,0.00),v(31.26,21.25,0.00),v(31.15,21.24,0.00),v(31.02,21.29,0.00),v(30.92,21.37,0.00),v(30.81,21.48,0.00),v(30.70,21.62,0.00),v(30.70,21.62,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(35.24,22.75,0.00),v(35.04,22.74,0.00),v(34.87,22.69,0.00),v(34.72,22.62,0.00),v(34.60,22.52,0.00),v(34.51,22.40,0.00),v(34.45,22.25,0.00),v(34.41,22.09,0.00),v(34.39,21.91,0.00),v(34.39,21.68,0.00),v(34.41,21.50,0.00),v(34.45,21.33,0.00),v(34.52,21.19,0.00),v(34.61,21.07,0.00),v(34.72,20.97,0.00),v(34.86,20.89,0.00),v(35.01,20.85,0.00),v(35.18,20.83,0.00),v(35.47,20.88,0.00),v(35.60,20.93,0.00),v(35.71,21.01,0.00),v(35.81,21.12,0.00),v(35.88,21.25,0.00),v(35.93,21.41,0.00),v(35.94,21.60,0.00),v(35.89,21.93,0.00),v(34.82,21.93,0.00),v(34.84,22.14,0.00),v(34.93,22.28,0.00),v(35.07,22.36,0.00),v(35.27,22.39,0.00),v(35.52,22.36,0.00),v(35.80,22.27,0.00),v(35.92,22.64,0.00),v(35.57,22.73,0.00),v(35.24,22.75,0.00),v(35.24,22.75,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(34.82,21.60,0.00),v(35.52,21.60,0.00),v(35.52,21.54,0.00),v(35.50,21.39,0.00),v(35.43,21.28,0.00),v(35.32,21.22,0.00),v(35.18,21.20,0.00),v(35.03,21.21,0.00),v(34.93,21.29,0.00),v(34.85,21.42,0.00),v(34.82,21.60,0.00),v(34.82,21.60,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(39.19,22.75,0.00),v(39.04,22.74,0.00),v(38.91,22.70,0.00),v(38.80,22.64,0.00),v(38.72,22.57,0.00),v(38.63,22.36,0.00),v(38.60,22.10,0.00),v(38.60,21.23,0.00),v(38.32,21.23,0.00),v(38.32,20.86,0.00),v(38.63,20.86,0.00),v(38.63,20.38,0.00),v(39.05,20.38,0.00),v(39.05,20.86,0.00),v(39.64,20.86,0.00),v(39.64,21.23,0.00),v(39.05,21.23,0.00),v(39.05,22.10,0.00),v(39.07,22.23,0.00),v(39.11,22.30,0.00),v(39.19,22.35,0.00),v(39.30,22.36,0.00),v(39.43,22.34,0.00),v(39.59,22.30,0.00),v(39.70,22.67,0.00),v(39.45,22.73,0.00),v(39.19,22.75,0.00),v(39.19,22.75,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(23.88,25.72,0.00),v(23.87,25.60,0.00),v(23.81,25.51,0.00),v(23.72,25.44,0.00),v(23.57,25.38,0.00),v(23.26,25.26,0.00),v(23.02,25.16,0.00),v(22.86,25.01,0.00),v(22.76,24.81,0.00),v(22.73,24.56,0.00),v(22.74,24.41,0.00),v(22.78,24.27,0.00),v(22.84,24.15,0.00),v(22.92,24.05,0.00),v(23.03,23.96,0.00),v(23.17,23.90,0.00),v(23.33,23.87,0.00),v(23.52,23.85,0.00),v(23.86,23.90,0.00),v(24.22,24.02,0.00),v(24.08,24.39,0.00),v(23.78,24.29,0.00),v(23.52,24.25,0.00),v(23.36,24.26,0.00),v(23.25,24.30,0.00),v(23.18,24.39,0.00),v(23.15,24.53,0.00),v(23.17,24.64,0.00),v(23.22,24.73,0.00),v(23.31,24.79,0.00),v(23.43,24.84,0.00),v(23.74,24.95,0.00),v(23.98,25.06,0.00),v(24.15,25.21,0.00),v(24.25,25.42,0.00),v(24.28,25.69,0.00),v(24.27,25.85,0.00),v(24.23,25.99,0.00),v(24.16,26.12,0.00),v(24.06,26.22,0.00),v(23.94,26.31,0.00),v(23.79,26.37,0.00),v(23.61,26.41,0.00),v(23.40,26.42,0.00),v(23.04,26.38,0.00),v(22.84,26.31,0.00),v(22.61,26.22,0.00),v(22.76,25.86,0.00),v(22.95,25.93,0.00),v(23.11,25.98,0.00),v(23.38,26.03,0.00),v(23.61,26.01,0.00),v(23.77,25.94,0.00),v(23.86,25.85,0.00),v(23.88,25.72,0.00),v(23.88,25.72,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(26.23,25.57,0.00),v(26.22,25.76,0.00),v(26.18,25.92,0.00),v(26.11,26.07,0.00),v(26.01,26.19,0.00),v(25.90,26.29,0.00),v(25.76,26.36,0.00),v(25.59,26.41,0.00),v(25.41,26.42,0.00),v(25.23,26.41,0.00),v(25.07,26.36,0.00),v(24.93,26.29,0.00),v(24.81,26.19,0.00),v(24.72,26.07,0.00),v(24.65,25.92,0.00),v(24.61,25.76,0.00),v(24.60,25.57,0.00),v(24.60,25.35,0.00),v(24.61,25.17,0.00),v(24.65,25.00,0.00),v(24.71,24.86,0.00),v(24.80,24.73,0.00),v(24.92,24.64,0.00),v(25.06,24.56,0.00),v(25.22,24.52,0.00),v(25.41,24.50,0.00),v(25.59,24.52,0.00),v(25.76,24.56,0.00),v(25.90,24.64,0.00),v(26.01,24.73,0.00),v(26.11,24.86,0.00),v(26.18,25.00,0.00),v(26.22,25.17,0.00),v(26.23,25.35,0.00),v(26.23,25.57,0.00),v(26.23,25.57,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(25.41,24.87,0.00),v(25.24,24.90,0.00),v(25.12,24.99,0.00),v(25.04,25.14,0.00),v(25.02,25.35,0.00),v(25.02,25.57,0.00),v(25.04,25.78,0.00),v(25.12,25.93,0.00),v(25.24,26.02,0.00),v(25.41,26.05,0.00),v(25.58,26.02,0.00),v(25.71,25.93,0.00),v(25.78,25.78,0.00),v(25.81,25.57,0.00),v(25.81,25.35,0.00),v(25.77,25.15,0.00),v(25.70,25.00,0.00),v(25.58,24.90,0.00),v(25.41,24.87,0.00),v(25.41,24.87,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(27.99,24.90,0.00),v(27.26,24.90,0.00),v(27.26,26.39,0.00),v(26.83,26.39,0.00),v(26.83,24.90,0.00),v(26.52,24.90,0.00),v(26.52,24.53,0.00),v(26.83,24.53,0.00),v(26.83,24.36,0.00),v(26.88,24.07,0.00),v(26.93,23.94,0.00),v(27.02,23.84,0.00),v(27.12,23.75,0.00),v(27.26,23.68,0.00),v(27.43,23.64,0.00),v(27.62,23.63,0.00),v(27.80,23.64,0.00),v(28.00,23.67,0.00),v(28.41,23.82,0.00),v(28.41,24.50,0.00),v(29.01,24.50,0.00),v(29.01,24.87,0.00),v(28.41,24.87,0.00),v(28.41,25.74,0.00),v(28.43,25.87,0.00),v(28.48,25.94,0.00),v(28.56,25.99,0.00),v(28.67,26.00,0.00),v(28.80,25.98,0.00),v(28.95,25.94,0.00),v(29.06,26.31,0.00),v(28.83,26.39,0.00),v(28.58,26.42,0.00),v(28.43,26.41,0.00),v(28.30,26.37,0.00),v(28.20,26.31,0.00),v(28.12,26.23,0.00),v(28.02,26.03,0.00),v(27.99,25.77,0.00),v(27.99,24.90,0.00),v(27.99,24.90,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(27.26,24.53,0.00),v(27.99,24.53,0.00),v(27.99,24.08,0.00),v(27.80,24.04,0.00),v(27.62,24.02,0.00),v(27.47,24.04,0.00),v(27.35,24.10,0.00),v(27.28,24.20,0.00),v(27.26,24.36,0.00),v(27.26,24.53,0.00),v(27.26,24.53,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(31.37,26.39,0.00),v(30.85,26.40,0.00),v(30.61,25.28,0.00),v(30.30,26.39,0.00),v(29.81,26.39,0.00),v(29.33,24.53,0.00),v(29.76,24.53,0.00),v(29.86,24.86,0.00),v(29.95,25.18,0.00),v(30.08,25.83,0.00),v(30.38,24.53,0.00),v(30.80,24.53,0.00),v(31.11,25.76,0.00),v(31.28,25.15,0.00),v(31.42,24.53,0.00),v(31.82,24.53,0.00),v(31.37,26.39,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(33.17,26.14,0.00),v(33.02,26.27,0.00),v(32.87,26.35,0.00),v(32.72,26.40,0.00),v(32.55,26.42,0.00),v(32.43,26.41,0.00),v(32.33,26.38,0.00),v(32.19,26.27,0.00),v(32.10,26.09,0.00),v(32.07,25.86,0.00),v(32.11,25.63,0.00),v(32.22,25.46,0.00),v(32.31,25.40,0.00),v(32.42,25.36,0.00),v(32.56,25.33,0.00),v(32.72,25.32,0.00),v(32.92,25.34,0.00),v(33.12,25.38,0.00),v(33.12,25.15,0.00),v(33.10,25.03,0.00),v(33.04,24.95,0.00),v(32.94,24.91,0.00),v(32.81,24.90,0.00),v(32.57,24.92,0.00),v(32.30,25.01,0.00),v(32.19,24.64,0.00),v(32.55,24.53,0.00),v(32.86,24.50,0.00),v(33.05,24.52,0.00),v(33.19,24.56,0.00),v(33.31,24.62,0.00),v(33.40,24.70,0.00),v(33.51,24.91,0.00),v(33.54,25.15,0.00),v(33.54,25.86,0.00),v(33.57,25.96,0.00),v(33.61,25.99,0.00),v(33.68,26.00,0.00),v(33.71,26.00,0.00),v(33.71,26.39,0.00),v(33.62,26.39,0.00),v(33.47,26.38,0.00),v(33.34,26.33,0.00),v(33.24,26.24,0.00),v(33.17,26.14,0.00),v(33.17,26.14,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(32.65,26.05,0.00),v(32.83,25.98,0.00),v(32.95,25.91,0.00),v(33.14,25.72,0.00),v(32.91,25.67,0.00),v(32.69,25.65,0.00),v(32.57,25.67,0.00),v(32.50,25.70,0.00),v(32.46,25.77,0.00),v(32.45,25.86,0.00),v(32.48,25.97,0.00),v(32.55,26.03,0.00),v(32.62,26.05,0.00),v(32.65,26.05,0.00),v(32.65,26.05,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(34.50,25.24,0.00),v(34.50,26.39,0.00),v(34.08,26.39,0.00),v(34.08,24.53,0.00),v(34.47,24.53,0.00),v(34.47,24.81,0.00),v(34.65,24.67,0.00),v(34.81,24.57,0.00),v(34.96,24.52,0.00),v(35.12,24.50,0.00),v(35.12,24.93,0.00),v(34.96,24.95,0.00),v(34.80,25.02,0.00),v(34.50,25.24,0.00),v(34.50,25.24,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(36.25,26.42,0.00),v(36.05,26.41,0.00),v(35.88,26.36,0.00),v(35.73,26.29,0.00),v(35.61,26.19,0.00),v(35.52,26.07,0.00),v(35.46,25.92,0.00),v(35.42,25.76,0.00),v(35.40,25.57,0.00),v(35.40,25.35,0.00),v(35.42,25.17,0.00),v(35.46,25.00,0.00),v(35.53,24.86,0.00),v(35.62,24.73,0.00),v(35.73,24.64,0.00),v(35.87,24.56,0.00),v(36.02,24.52,0.00),v(36.19,24.50,0.00),v(36.48,24.55,0.00),v(36.61,24.60,0.00),v(36.72,24.68,0.00),v(36.82,24.79,0.00),v(36.89,24.92,0.00),v(36.94,25.08,0.00),v(36.95,25.26,0.00),v(36.90,25.60,0.00),v(35.83,25.60,0.00),v(35.85,25.80,0.00),v(35.94,25.94,0.00),v(36.08,26.03,0.00),v(36.28,26.05,0.00),v(36.54,26.03,0.00),v(36.81,25.94,0.00),v(36.93,26.31,0.00),v(36.57,26.40,0.00),v(36.25,26.42,0.00),v(36.25,26.42,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(35.83,25.26,0.00),v(36.53,25.26,0.00),v(36.53,25.21,0.00),v(36.51,25.06,0.00),v(36.44,24.95,0.00),v(36.33,24.89,0.00),v(36.19,24.87,0.00),v(36.04,24.88,0.00),v(35.92,24.96,0.00),v(35.85,25.09,0.00),v(35.83,25.26,0.00),v(35.83,25.26,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(24.52,28.61,0.00),v(23.59,28.61,0.00),v(23.59,29.66,0.00),v(23.13,29.66,0.00),v(23.13,27.18,0.00),v(23.59,27.18,0.00),v(23.59,28.19,0.00),v(24.52,28.19,0.00),v(24.52,27.18,0.00),v(24.97,27.18,0.00),v(24.97,29.66,0.00),v(24.52,29.66,0.00),v(24.52,28.61,0.00),v(24.52,28.61,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(26.32,29.69,0.00),v(26.13,29.67,0.00),v(25.95,29.63,0.00),v(25.81,29.55,0.00),v(25.69,29.45,0.00),v(25.60,29.33,0.00),v(25.53,29.19,0.00),v(25.49,29.02,0.00),v(25.48,28.84,0.00),v(25.48,28.61,0.00),v(25.49,28.43,0.00),v(25.53,28.27,0.00),v(25.60,28.12,0.00),v(25.69,28.00,0.00),v(25.81,27.90,0.00),v(25.94,27.83,0.00),v(26.10,27.78,0.00),v(26.27,27.77,0.00),v(26.55,27.81,0.00),v(26.68,27.87,0.00),v(26.80,27.95,0.00),v(26.89,28.05,0.00),v(26.97,28.18,0.00),v(27.01,28.34,0.00),v(27.03,28.53,0.00),v(26.97,28.87,0.00),v(25.87,28.87,0.00),v(25.90,29.07,0.00),v(25.98,29.21,0.00),v(26.12,29.29,0.00),v(26.32,29.32,0.00),v(26.58,29.30,0.00),v(26.86,29.21,0.00),v(26.97,29.57,0.00),v(26.64,29.66,0.00),v(26.32,29.69,0.00),v(26.32,29.69,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(25.90,28.53,0.00),v(26.61,28.53,0.00),v(26.61,28.47,0.00),v(26.58,28.33,0.00),v(26.51,28.22,0.00),v(26.40,28.16,0.00),v(26.27,28.13,0.00),v(26.12,28.15,0.00),v(26.00,28.23,0.00),v(25.93,28.36,0.00),v(25.90,28.53,0.00),v(25.90,28.53,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(27.85,28.50,0.00),v(27.85,29.66,0.00),v(27.43,29.66,0.00),v(27.43,27.80,0.00),v(27.82,27.80,0.00),v(27.82,28.08,0.00),v(28.00,27.94,0.00),v(28.16,27.84,0.00),v(28.31,27.79,0.00),v(28.47,27.77,0.00),v(28.47,28.20,0.00),v(28.31,28.22,0.00),v(28.15,28.28,0.00),v(27.85,28.50,0.00),v(27.85,28.50,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(28.89,27.46,0.00),v(28.89,27.03,0.00),v(29.34,27.03,0.00),v(29.34,27.46,0.00),v(28.89,27.46,0.00),v(28.89,27.46,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(28.89,29.66,0.00),v(28.89,27.80,0.00),v(29.32,27.80,0.00),v(29.32,29.66,0.00),v(28.89,29.66,0.00),v(28.89,29.66,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(30.63,29.69,0.00),v(30.47,29.67,0.00),v(30.35,29.64,0.00),v(30.24,29.58,0.00),v(30.16,29.50,0.00),v(30.07,29.29,0.00),v(30.04,29.04,0.00),v(30.04,28.16,0.00),v(29.73,28.16,0.00),v(29.73,27.80,0.00),v(30.04,27.80,0.00),v(30.04,27.32,0.00),v(30.46,27.32,0.00),v(30.46,27.80,0.00),v(31.05,27.80,0.00),v(31.05,28.16,0.00),v(30.46,28.16,0.00),v(30.46,29.04,0.00),v(30.48,29.16,0.00),v(30.52,29.24,0.00),v(30.60,29.28,0.00),v(30.71,29.29,0.00),v(30.84,29.27,0.00),v(31.00,29.24,0.00),v(31.11,29.60,0.00),v(30.88,29.67,0.00),v(30.63,29.69,0.00),v(30.63,29.69,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(32.45,29.41,0.00),v(32.30,29.54,0.00),v(32.15,29.63,0.00),v(31.99,29.68,0.00),v(31.83,29.70,0.00),v(31.71,29.69,0.00),v(31.61,29.66,0.00),v(31.46,29.54,0.00),v(31.38,29.36,0.00),v(31.35,29.13,0.00),v(31.38,28.90,0.00),v(31.49,28.74,0.00),v(31.58,28.68,0.00),v(31.69,28.63,0.00),v(31.83,28.60,0.00),v(32.00,28.60,0.00),v(32.20,28.61,0.00),v(32.39,28.65,0.00),v(32.39,28.43,0.00),v(32.37,28.30,0.00),v(32.31,28.22,0.00),v(32.21,28.18,0.00),v(32.08,28.17,0.00),v(31.85,28.20,0.00),v(31.57,28.28,0.00),v(31.46,27.92,0.00),v(31.82,27.80,0.00),v(32.14,27.78,0.00),v(32.32,27.79,0.00),v(32.47,27.83,0.00),v(32.59,27.89,0.00),v(32.68,27.97,0.00),v(32.79,28.18,0.00),v(32.82,28.43,0.00),v(32.82,29.13,0.00),v(32.84,29.23,0.00),v(32.89,29.26,0.00),v(32.96,29.27,0.00),v(32.99,29.27,0.00),v(32.99,29.67,0.00),v(32.90,29.67,0.00),v(32.73,29.66,0.00),v(32.60,29.60,0.00),v(32.51,29.52,0.00),v(32.45,29.41,0.00),v(32.45,29.41,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(34.03,30.49,0.00),v(33.60,30.44,0.00),v(33.38,30.38,0.00),v(33.18,30.29,0.00),v(33.30,29.95,0.00),v(33.66,30.08,0.00),v(34.00,30.12,0.00),v(34.21,30.10,0.00),v(34.36,30.06,0.00),v(34.45,29.99,0.00),v(34.48,29.89,0.00),v(34.47,29.82,0.00),v(34.44,29.77,0.00),v(34.37,29.73,0.00),v(34.28,29.72,0.00),v(33.83,29.72,0.00),v(33.60,29.69,0.00),v(33.44,29.60,0.00),v(33.35,29.47,0.00),v(33.32,29.33,0.00),v(33.37,29.13,0.00),v(33.49,28.96,0.00),v(33.38,28.87,0.00),v(33.29,28.75,0.00),v(33.23,28.60,0.00),v(33.21,28.43,0.00),v(33.23,28.28,0.00),v(33.27,28.15,0.00),v(33.33,28.03,0.00),v(33.42,27.94,0.00),v(33.64,27.82,0.00),v(33.92,27.78,0.00),v(33.97,27.78,0.00),v(34.11,27.78,0.00),v(34.26,27.80,0.00),v(34.93,27.80,0.00),v(34.93,28.17,0.00),v(34.57,28.12,0.00),v(34.65,28.27,0.00),v(34.68,28.48,0.00),v(34.67,28.61,0.00),v(34.63,28.73,0.00),v(34.48,28.92,0.00),v(34.26,29.03,0.00),v(34.00,29.07,0.00),v(33.95,29.07,0.00),v(33.87,29.07,0.00),v(33.80,29.07,0.00),v(33.74,29.16,0.00),v(33.72,29.24,0.00),v(33.76,29.35,0.00),v(33.81,29.38,0.00),v(33.89,29.39,0.00),v(34.34,29.39,0.00),v(34.57,29.42,0.00),v(34.75,29.51,0.00),v(34.86,29.67,0.00),v(34.91,29.89,0.00),v(34.89,30.02,0.00),v(34.84,30.14,0.00),v(34.77,30.24,0.00),v(34.67,30.33,0.00),v(34.54,30.40,0.00),v(34.39,30.44,0.00),v(34.22,30.48,0.00),v(34.03,30.49,0.00),v(34.03,30.49,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(34.28,28.43,0.00),v(34.26,28.29,0.00),v(34.20,28.20,0.00),v(34.11,28.14,0.00),v(34.00,28.12,0.00),v(33.95,28.12,0.00),v(33.83,28.14,0.00),v(33.74,28.20,0.00),v(33.68,28.29,0.00),v(33.66,28.43,0.00),v(33.69,28.56,0.00),v(33.75,28.66,0.00),v(33.84,28.72,0.00),v(33.95,28.74,0.00),v(34.00,28.74,0.00),v(34.11,28.70,0.00),v(34.20,28.64,0.00),v(34.26,28.55,0.00),v(34.28,28.43,0.00),v(34.28,28.43,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(36.03,29.70,0.00),v(35.84,29.68,0.00),v(35.66,29.63,0.00),v(35.52,29.56,0.00),v(35.40,29.46,0.00),v(35.31,29.34,0.00),v(35.24,29.20,0.00),v(35.20,29.03,0.00),v(35.19,28.85,0.00),v(35.19,28.62,0.00),v(35.20,28.44,0.00),v(35.24,28.28,0.00),v(35.31,28.13,0.00),v(35.40,28.01,0.00),v(35.52,27.91,0.00),v(35.65,27.84,0.00),v(35.81,27.79,0.00),v(35.98,27.78,0.00),v(36.26,27.82,0.00),v(36.39,27.88,0.00),v(36.51,27.96,0.00),v(36.60,28.06,0.00),v(36.68,28.19,0.00),v(36.72,28.35,0.00),v(36.74,28.54,0.00),v(36.68,28.88,0.00),v(35.61,28.88,0.00),v(35.64,29.08,0.00),v(35.72,29.22,0.00),v(35.86,29.30,0.00),v(36.06,29.33,0.00),v(36.32,29.30,0.00),v(36.60,29.22,0.00),v(36.71,29.58,0.00),v(36.35,29.67,0.00),v(36.03,29.70,0.00),v(36.03,29.70,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(35.61,28.54,0.00),v(36.32,28.54,0.00),v(36.32,28.48,0.00),v(36.29,28.33,0.00),v(36.22,28.23,0.00),v(36.11,28.16,0.00),v(35.98,28.14,0.00),v(35.83,28.16,0.00),v(35.72,28.24,0.00),v(35.65,28.37,0.00),v(35.61,28.54,0.00),v(35.61,28.54,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(32.77,21.63,0.00),v(32.77,22.73,0.00),v(32.35,22.73,0.00),v(32.35,20.87,0.00),v(32.74,20.87,0.00),v(32.74,21.12,0.00),v(32.91,20.99,0.00),v(33.07,20.90,0.00),v(33.22,20.85,0.00),v(33.39,20.84,0.00),v(33.52,20.85,0.00),v(33.63,20.88,0.00),v(33.72,20.93,0.00),v(33.80,21.00,0.00),v(33.89,21.21,0.00),v(33.93,21.49,0.00),v(33.93,22.73,0.00),v(33.50,22.73,0.00),v(33.50,21.54,0.00),v(33.48,21.41,0.00),v(33.43,21.31,0.00),v(33.34,21.25,0.00),v(33.22,21.25,0.00),v(33.10,21.29,0.00),v(32.99,21.37,0.00),v(32.89,21.48,0.00),v(32.77,21.63,0.00),v(32.77,21.63,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(36.83,21.64,0.00),v(36.83,22.75,0.00),v(36.40,22.75,0.00),v(36.40,20.88,0.00),v(36.80,20.88,0.00),v(36.80,21.14,0.00),v(36.96,21.01,0.00),v(37.12,20.92,0.00),v(37.28,20.87,0.00),v(37.45,20.86,0.00),v(37.58,20.87,0.00),v(37.69,20.90,0.00),v(37.78,20.95,0.00),v(37.85,21.02,0.00),v(37.95,21.23,0.00),v(37.98,21.51,0.00),v(37.98,22.75,0.00),v(37.56,22.75,0.00),v(37.56,21.56,0.00),v(37.54,21.43,0.00),v(37.48,21.33,0.00),v(37.39,21.27,0.00),v(37.28,21.26,0.00),v(37.15,21.31,0.00),v(37.05,21.39,0.00),v(36.94,21.50,0.00),v(36.83,21.64,0.00),v(36.83,21.64,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(31.92,29.33,0.00),v(32.09,29.27,0.00),v(32.22,29.19,0.00),v(32.41,29.00,0.00),v(32.18,28.96,0.00),v(31.96,28.93,0.00),v(31.84,28.95,0.00),v(31.77,28.98,0.00),v(31.73,29.05,0.00),v(31.72,29.14,0.00),v(31.75,29.26,0.00),v(31.82,29.31,0.00),v(31.89,29.33,0.00),v(31.92,29.33,0.00),v(31.92,29.33,0.00),})),}
dt = translate(- v(11.30,15.84,0.00)) * merge{(linear_extrude_from_oriented(v(0,0,1),{v(11.30,15.84,0.00),v(10.98,15.81,0.00),v(10.63,15.69,0.00),v(10.75,15.35,0.00),v(10.91,15.41,0.00),v(11.05,15.45,0.00),v(11.29,15.48,0.00),v(11.44,15.46,0.00),v(11.55,15.41,0.00),v(11.62,15.32,0.00),v(11.64,15.19,0.00),v(11.64,15.14,0.00),v(11.62,15.03,0.00),v(11.56,14.94,0.00),v(11.46,14.89,0.00),v(11.33,14.87,0.00),v(10.98,14.87,0.00),v(10.98,14.51,0.00),v(11.33,14.51,0.00),v(11.46,14.49,0.00),v(11.55,14.44,0.00),v(11.60,14.36,0.00),v(11.62,14.25,0.00),v(11.62,14.20,0.00),v(11.60,14.07,0.00),v(11.54,13.98,0.00),v(11.44,13.93,0.00),v(11.30,13.92,0.00),v(11.06,13.95,0.00),v(10.78,14.04,0.00),v(10.66,13.70,0.00),v(11.00,13.59,0.00),v(11.31,13.55,0.00),v(11.48,13.57,0.00),v(11.63,13.60,0.00),v(11.75,13.66,0.00),v(11.85,13.74,0.00),v(11.97,13.94,0.00),v(12.01,14.19,0.00),v(12.01,14.23,0.00),v(11.99,14.38,0.00),v(11.93,14.50,0.00),v(11.84,14.61,0.00),v(11.73,14.69,0.00),v(11.85,14.77,0.00),v(11.94,14.88,0.00),v(12.00,15.01,0.00),v(12.02,15.16,0.00),v(12.02,15.21,0.00),v(11.99,15.45,0.00),v(11.86,15.66,0.00),v(11.76,15.73,0.00),v(11.63,15.79,0.00),v(11.48,15.83,0.00),v(11.30,15.84,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(13.76,15.59,0.00),v(13.62,15.70,0.00),v(13.49,15.78,0.00),v(13.35,15.83,0.00),v(13.21,15.84,0.00),v(13.02,15.81,0.00),v(12.88,15.70,0.00),v(12.81,15.55,0.00),v(12.78,15.35,0.00),v(12.81,15.15,0.00),v(12.91,14.99,0.00),v(13.08,14.89,0.00),v(13.20,14.87,0.00),v(13.35,14.86,0.00),v(13.71,14.90,0.00),v(13.71,14.71,0.00),v(13.69,14.59,0.00),v(13.64,14.52,0.00),v(13.55,14.48,0.00),v(13.44,14.47,0.00),v(13.22,14.49,0.00),v(12.97,14.57,0.00),v(12.86,14.25,0.00),v(13.18,14.15,0.00),v(13.46,14.12,0.00),v(13.63,14.13,0.00),v(13.76,14.17,0.00),v(13.87,14.22,0.00),v(13.95,14.30,0.00),v(14.05,14.48,0.00),v(14.07,14.71,0.00),v(14.07,15.34,0.00),v(14.10,15.44,0.00),v(14.14,15.47,0.00),v(14.21,15.48,0.00),v(14.25,15.48,0.00),v(14.25,15.83,0.00),v(14.17,15.83,0.00),v(14.02,15.81,0.00),v(13.90,15.76,0.00),v(13.82,15.68,0.00),v(13.76,15.59,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(13.33,15.50,0.00),v(13.40,15.49,0.00),v(13.48,15.46,0.00),v(13.58,15.38,0.00),v(13.71,15.27,0.00),v(13.71,15.20,0.00),v(13.54,15.16,0.00),v(13.38,15.14,0.00),v(13.27,15.15,0.00),v(13.20,15.19,0.00),v(13.16,15.25,0.00),v(13.14,15.34,0.00),v(13.15,15.41,0.00),v(13.18,15.46,0.00),v(13.30,15.50,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(15.19,15.81,0.00),v(14.84,15.81,0.00),v(14.25,14.15,0.00),v(14.63,14.15,0.00),v(14.92,14.98,0.00),v(15.03,15.32,0.00),v(15.13,14.98,0.00),v(15.40,14.15,0.00),v(15.78,14.15,0.00),v(15.19,15.81,0.00),v(15.19,15.81,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(16.42,15.81,0.00),v(16.04,15.81,0.00),v(16.04,14.15,0.00),v(16.40,14.15,0.00),v(16.40,14.41,0.00),v(16.55,14.27,0.00),v(16.70,14.19,0.00),v(16.83,14.14,0.00),v(16.98,14.12,0.00),v(16.98,14.50,0.00),v(16.83,14.52,0.00),v(16.68,14.59,0.00),v(16.42,14.79,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(17.28,14.14,0.00),v(17.66,14.14,0.00),v(17.66,15.80,0.00),v(17.28,15.80,0.00),v(17.28,14.14,0.00),v(17.28,14.14,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(17.27,13.44,0.00),v(17.67,13.44,0.00),v(17.67,13.83,0.00),v(17.27,13.83,0.00),v(17.27,13.44,0.00),v(17.27,13.44,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(18.09,13.38,0.00),v(18.46,13.38,0.00),v(18.46,15.33,0.00),v(18.49,15.43,0.00),v(18.52,15.46,0.00),v(18.58,15.47,0.00),v(18.64,15.47,0.00),v(18.64,15.82,0.00),v(18.54,15.82,0.00),v(18.42,15.81,0.00),v(18.32,15.78,0.00),v(18.18,15.66,0.00),v(18.11,15.51,0.00),v(18.09,15.33,0.00),v(18.09,13.38,0.00),v(18.09,13.38,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(19.37,15.80,0.00),v(19.37,15.49,0.00),v(19.38,15.33,0.00),v(19.40,15.18,0.00),v(19.50,14.95,0.00),v(19.65,14.79,0.00),v(19.83,14.67,0.00),v(20.05,14.56,0.00),v(20.26,14.42,0.00),v(20.32,14.33,0.00),v(20.34,14.23,0.00),v(20.34,14.17,0.00),v(20.32,14.06,0.00),v(20.25,13.98,0.00),v(20.14,13.93,0.00),v(19.99,13.91,0.00),v(19.76,13.94,0.00),v(19.49,14.03,0.00),v(19.37,13.70,0.00),v(19.70,13.58,0.00),v(20.00,13.55,0.00),v(20.18,13.56,0.00),v(20.34,13.59,0.00),v(20.46,13.65,0.00),v(20.56,13.73,0.00),v(20.68,13.93,0.00),v(20.72,14.17,0.00),v(20.72,14.23,0.00),v(20.68,14.46,0.00),v(20.57,14.64,0.00),v(20.41,14.78,0.00),v(20.22,14.89,0.00),v(20.00,15.00,0.00),v(19.89,15.07,0.00),v(19.82,15.16,0.00),v(19.77,15.28,0.00),v(19.75,15.43,0.00),v(20.72,15.43,0.00),v(20.72,15.80,0.00),v(19.37,15.80,0.00),v(19.37,15.80,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(21.80,13.91,0.00),v(21.63,13.94,0.00),v(21.51,14.05,0.00),v(21.44,14.22,0.00),v(21.41,14.45,0.00),v(21.41,14.92,0.00),v(21.44,15.16,0.00),v(21.51,15.33,0.00),v(21.63,15.44,0.00),v(21.80,15.47,0.00),v(21.98,15.44,0.00),v(22.10,15.33,0.00),v(22.17,15.16,0.00),v(22.19,14.92,0.00),v(22.19,14.45,0.00),v(22.17,14.22,0.00),v(22.10,14.05,0.00),v(21.98,13.94,0.00),v(21.80,13.91,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(22.59,14.92,0.00),v(22.58,15.12,0.00),v(22.54,15.30,0.00),v(22.48,15.46,0.00),v(22.39,15.59,0.00),v(22.28,15.70,0.00),v(22.14,15.77,0.00),v(21.99,15.82,0.00),v(21.80,15.83,0.00),v(21.62,15.82,0.00),v(21.46,15.77,0.00),v(21.33,15.70,0.00),v(21.22,15.59,0.00),v(21.13,15.46,0.00),v(21.07,15.30,0.00),v(21.03,15.12,0.00),v(21.02,14.92,0.00),v(21.02,14.45,0.00),v(21.03,14.25,0.00),v(21.07,14.07,0.00),v(21.13,13.92,0.00),v(21.22,13.79,0.00),v(21.33,13.68,0.00),v(21.46,13.61,0.00),v(21.62,13.56,0.00),v(21.80,13.54,0.00),v(21.99,13.56,0.00),v(22.14,13.61,0.00),v(22.28,13.68,0.00),v(22.39,13.79,0.00),v(22.48,13.92,0.00),v(22.54,14.07,0.00),v(22.58,14.25,0.00),v(22.59,14.45,0.00),v(22.59,14.92,0.00),v(22.59,14.92,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(23.34,13.58,0.00),v(23.67,13.58,0.00),v(23.67,15.80,0.00),v(23.28,15.80,0.00),v(23.28,14.01,0.00),v(22.95,14.19,0.00),v(22.77,13.89,0.00),v(23.34,13.58,0.00),v(23.34,13.58,0.00),})),(linear_extrude_from_oriented(v(0,0,1),{v(24.72,15.80,0.00),v(24.30,15.80,0.00),v(24.55,14.87,0.00),v(24.92,13.95,0.00),v(23.96,13.95,0.00),v(23.96,13.58,0.00),v(25.34,13.58,0.00),v(25.34,13.91,0.00),v(24.96,14.86,0.00),v(24.72,15.80,0.00),})),}
end

---------------------------------------------------

-- carves the base with all text/logos
carve = 1.5
angle_align = 146.4
base = difference{

   translate(3,0,0)*rotate(-90,0,0)*base,

   translate(0,0,0)
  *rotate(angle_align+180,0,0)
  *translate(-38.7,8.8,41.5-carve)
  *scale(1.27)*translate(-11.2,22.7,0)*scale(1.1)*mirror(Y)*scale(1,1,10)*unesco,

   rotate(angle_align,-60,0)
  *translate(-21.9,4.7,-49.3+carve)
  *scale(0.32,0.32,10)*inria,

   rotate(angle_align,-240,0)
  *translate(-17.2,10.4,-49.6+carve)
  *scale(2.7,2.7,6)*fh,

   rotate(angle_align,-180,0) 
  *translate(-11.8,5.3,-47.5+carve)
  *scale(2.7,2.7,6)*ib,

   rotate(angle_align,-300,0)
  *translate(-16.6,17.6,-49.6+carve)
  *scale(2.7,2.7,6)*dt,
}

T = rotate(angle_align,-120,0)
 *translate(-20.2,7.4,-45.3+carve)
 *scale(2.1,2.1,6)
base = difference{base,
 T * so
}

---------------------------------------------------
-- final assembly

final = union{
   difference(scale(1.3)*final,translate(-2,-21,0)*rotate(0,30,0)*translate(3,0,0)*rotate(-90,0,0)*base_carve),
   translate(-2,-21,0)*rotate(0,30,0)*base
}
final = rotate(90,0,0)*final

---------------------------------------------------
-- either emit for display or outputs a mesh

if not do_mesh then
  emit(final)
else
  m = to_mesh(final,0.2)
  dump(m,Path .. 'sculpture.stl')
  emit(m)
end

---------------------------------------------------
